/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/dashboardtypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class IModule;
class IModuleListener;

class ModuleBase;
class ComponentContainer;

/**
 * Allows clients to quickly shut down all modules in the SDK synchronously.  The list of modules must be given in
 * order of shutdown.  Thus, CoreAPI must appear last in the list if it is to be shut down.
 */
void ShutdownModulesSync(const std::vector<std::shared_ptr<ttv::IModule>>& modules);
}  // namespace ttv

/**
 * The interface to be implemented by all modules that exist in the Twitch SDK.
 *
 * Clients should not consider the module to be thread safe unless a method explicitly says it is.  Thus, all calls into
 * a module should be made on the same thread.
 */
class ttv::IModule {
 public:
  enum class State { Uninitialized, Initializing, Initialized, ShuttingDown };

  using InitializeCallback = std::function<void(TTV_ErrorCode ec)>;
  using ShutdownCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  virtual ~IModule() = default;

  /**
   * Returns the current state of the module.
   */
  virtual State GetState() const = 0;
  /**
   * Returns the name of the module.
   */
  virtual std::string GetModuleName() const = 0;
  /**
   * Initializes the module asynchronously.  The module state will be reported in IModuleListener::ModuleStateChanged().
   * No other calls should be made into the module until the module state is either State::Initialized or
   * State::Uninitialized. You should only expect changes to the module state if this method succeeds.
   */
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) = 0;
  /**
   * Updates the internal state of the module and fires any pending callbacks.
   */
  virtual TTV_ErrorCode Update() = 0;
  /**
   * Initiates an asynchronous shutdown of the module.  The module state will be reported in
   * IModuleListener::ModuleStateChanged(). No other calls should be made into the module until the module state is
   * either State::Initialized or State::Uninitialized. All modules that depend on this module need to be shutdown
   * first. You should only expect changes to the module state if this method succeeds.
   */
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) = 0;
};

/**
 * A base class providing common functionality for modules.
 */
class ttv::ModuleBase : public ttv::IModule {
 public:
  ModuleBase();

  // IModule implementation
  virtual State GetState() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Update() override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;

 protected:
  void NotifyStateChange();
  void RegisterInitializeCallback(const InitializeCallback& callback) { mInitializeCallbacks.Push(callback); }

  void RegisterShutdownCallback(const ShutdownCallback& callback) { mShutdownCallbacks.Push(callback); }

  virtual bool CheckShutdown();
  virtual void CompleteShutdown();

  /**
   * Invokes the callback for the module listeners.
   */
  template <typename T>
  void Invoke(std::function<void(std::shared_ptr<T>)> callback) {
    mListeners.Invoke(
      [callback](std::shared_ptr<IModuleListener> listener) { callback(std::static_pointer_cast<T>(listener)); });
  }

  /**
   * Returns the common ComponentContainer which hosts all components which live at the module level.
   */
  std::shared_ptr<ComponentContainer> GetComponentContainer() { return mComponentContainer; }

 protected:
  /**
   * The registered listeners.  These listeners extend IModuleListener and will be of the type required for the module.
   */
  EventSource<IModuleListener> mListeners;
  State mState;
  State mLastReportedState;

 private:
  /**
   * The set of components that are managed at the module level.  This may contain components which have been passed to
   * the client via the public interface.
   */
  std::shared_ptr<ComponentContainer> mComponentContainer;

  CallbackQueue<InitializeCallback> mInitializeCallbacks;
  CallbackQueue<ShutdownCallback> mShutdownCallbacks;
};

/**
 * The base interface for events coming from modules.  Modules should support a listener interface that extends this
 * one. These callbacks will be fired by the module when the client calls IModule::Update() but they may be called from
 * within other methods as well if they make some immediate change to the module state.
 */
class ttv::IModuleListener {
 public:
  /**
   * Called when the internal state of the module changes.  This usually happens because of calls to
   * IModule::Initialize() and IModule::Shutdown().
   */
  virtual void ModuleStateChanged(IModule* source, IModule::State state, TTV_ErrorCode ec) = 0;
  virtual ~IModuleListener() = default;
};
