/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <memory>

#ifdef CreateMutex
#undef CreateMutex
#endif

namespace ttv {
class IMutex;
class IConditionMutex;
class IMutexFactory;
class AutoMutex;

void SetMutexFactory(std::shared_ptr<IMutexFactory> factory);

TTV_ErrorCode CreateMutex(std::unique_ptr<ttv::IMutex>& result, const std::string& name = "");
TTV_ErrorCode CreateConditionMutex(std::unique_ptr<ttv::IConditionMutex>& resultresult, const std::string& name = "");
}  // namespace ttv

/**
 * Interface for basic mutex lock/unlock functionality.
 */
class ttv::IMutex {
 public:
  virtual ~IMutex() = default;

  /**
   * Block/wait until the mutex is free to take, and then acquire the mutex.
   * @return
   *   - TTV_EC_SUCCESS: Mutex has been successfully acquired.
   */
  virtual TTV_ErrorCode Lock() = 0;

  /**
   * Attempt to acquire the mutex, without waiting for the lock to be available.
   * @return
   *   - TTV_EC_SUCCESS: Mutex has been successfully acquired.
   *   - TTV_EC_MUTEX_NOT_ACQUIRED: Lock has failed and mutex has not been acquired.
   */
  virtual TTV_ErrorCode TryLock() = 0;

  /**
   * Release the previously acquired mutex (through Lock()).
   * @return
   *   - TTV_EC_SUCCESS: Mutex has been sucessfully released.
   */
  virtual TTV_ErrorCode Unlock() = 0;
};

/* Interface for a mutex with condition variable functionality.
 *
 * Wait() and WaitFor() must be called under lock (i.e. after Lock())
 * Signal() and Broadcast() in general should be called outside of lock to avoid pessimization.
 */
class ttv::IConditionMutex : public IMutex {
 public:
  virtual ~IConditionMutex() = default;

  /**
   * Wait on the IConditionMutex until notified to wake up (through Signal()/Broadcast()).
   * @return
   *   - TTV_EC_SUCCESS: Waited successfully on the IConditionMutex.
   *   - TTV_EC_CONDITION_WAIT_FAILED: Wait was not successful.
   */
  virtual TTV_ErrorCode Wait() = 0;

  /**
   * Wait on the IConditionMutex until notified to wake up OR the timeout is reached.
   * @param[in] timeoutMilliseconds The timeout to wait in milliseconds.
   * @return
   *   - TTV_EC_SUCCESS: Waited sucessfully on the IConditionMutex (and was notified through Signal()/Broadcast()).
   *   - TTV_EC_WAIT_TIMEOUT: The timeout was hit while waiting on IConditionMutex.
   *   - TTV_EC_CONDITION_WAIT_FAILED: Wait was not successful.
   */
  virtual TTV_ErrorCode WaitFor(uint64_t timeoutMilliseconds) = 0;

  /* Notify one thread waiting to be notified (through Wait()/WaitFor()).
   * Nothing happens if there are no threads waiting.
   * @return
   *   - TTV_EC_SUCCESS: Successfully notified a waiting thread (or there were no waiting threads).
   */
  virtual TTV_ErrorCode Signal() = 0;

  /* Notify all threads waiting to be notified (through Wait()/WaitFor()).
   * Nothing happens if there are no threads waiting.
   * @return
   *   - TTV_EC_SUCCESS: Successfully notified all waiting threads (or there were no waiting threads).
   */
  virtual TTV_ErrorCode Broadcast() = 0;
};

class ttv::IMutexFactory {
 public:
  virtual ~IMutexFactory() = default;

  /*
   * Create a mutex and return the newly created mutex in result. Optionally, the mutex can be given a name
   * through name.
   * @param[out] The created mutex.
   * @param[in] The name to give to the mutex
   * @return
   *   - TTV_EC_SUCCESS: Mutex was successfully created.
   */
  virtual TTV_ErrorCode CreateMutex(std::unique_ptr<ttv::IMutex>& result, const std::string& name = "") = 0;

  /*
   * Create a condition mutex and return the newly created condition mutex in result. Optionally, the condition mutex
   * can be given a name through name.
   * @param[out] The created condition mutex.
   * @param[in] The name to give to the condition mutex
   * @return
   *   - TTV_EC_SUCCESS: Mutex was successfully created.
   */
  virtual TTV_ErrorCode CreateConditionMutex(
    std::unique_ptr<ttv::IConditionMutex>& result, const std::string& name = "") = 0;
};

class ttv::AutoMutex {
 public:
  explicit AutoMutex(IMutex* mutex);
  ~AutoMutex();

 private:
  IMutex* mMutex;
};
