/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/variant.h"

#include <functional>
#include <stdexcept>

namespace ttv {
// An empty type for use as an empty alternative in std::variant.
struct Monostate;

template <typename ValueType>
class Optional;
}  // namespace ttv

/**
 * // A class that contains an optional value.
 * // @tparam ValueType The type of the value the optional class holds. The type may not be const, volatile,
 * //                     or a reference.
 * template <typename ValueType>
 * class ttv::Optional
 * {
 * public:
 *     // Constructs the optional without a value.
 *     Optional();
 *
 *     // Constructs the optional with a value.
 *     Optional(const ValueType& value);
 *     Optional(ValueType&& value);
 *
 *     // Returns true if the optional contains a valid value.
 *     bool HasValue() const;
 *     explicit operator bool() const;
 *
 *     // Returns the value contained by the optional.
 *     // It is illegal to call these methods when the optional does not contain a valid value.
 *     // It is the responsibility of the clients to check if the optional contains a value first before accessing the
 * value. ValueType& Value() &; const ValueType& Value() const &; ValueType& operator*(); const ValueType& operator()
 * const;
 *
 *     // Removes the value from optional if it had a value.
 *     void Clear();
 * };
 */

namespace ttv {
struct Monostate {};
constexpr bool operator<(Monostate, Monostate) noexcept {
  return false;
}
constexpr bool operator>(Monostate, Monostate) noexcept {
  return false;
}
constexpr bool operator<=(Monostate, Monostate) noexcept {
  return true;
}
constexpr bool operator>=(Monostate, Monostate) noexcept {
  return true;
}
constexpr bool operator==(Monostate, Monostate) noexcept {
  return true;
}
constexpr bool operator!=(Monostate, Monostate) noexcept {
  return false;
}

template <typename ValueType>
class Optional {
 public:
  constexpr Optional() noexcept : mContainer(Monostate{}) {}

  constexpr Optional(const ValueType& value) : mContainer(value) {}

  constexpr Optional(ValueType&& value) : mContainer(std::move(value)) {}

  Optional& operator=(ValueType&& arg) {
    mContainer = std::move(arg);
    return *this;
  }

  bool HasValue() const { return mContainer.template Is<ValueType>(); }

  explicit operator bool() const { return mContainer.template Is<ValueType>(); }

  ValueType& Value() { return mContainer.template As<ValueType>(); }

  const ValueType& Value() const { return mContainer.template As<ValueType>(); }

  ValueType& operator*() { return mContainer.template As<ValueType>(); }

  const ValueType& operator*() const { return mContainer.template As<ValueType>(); }

  // Note that this is not as performant as possible if called with lvalues since it introduces an extra copy
  ValueType ValueOrDefault(ValueType defaultValue) const {
    if (HasValue()) {
      return Value();
    } else {
      return std::move(defaultValue);
    }
  }

  void Clear() { mContainer = Monostate{}; }

 private:
  Variant<Monostate, ValueType> mContainer;
};

// Comparisons between optionals
template <typename ValueType>
bool operator==(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  if (lhs.HasValue() != rhs.HasValue()) {
    return false;
  }
  if (!lhs.HasValue()) {
    return true;
  }
  return *lhs == *rhs;
}

template <typename ValueType>
bool operator!=(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  if (lhs.HasValue() != rhs.HasValue()) {
    return true;
  }
  if (!lhs.HasValue()) {
    return false;
  }
  return *lhs != *rhs;
}

template <typename ValueType>
bool operator<(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  // No value is defined to be less than any existing value.
  if (!rhs.HasValue()) {
    return false;
  }
  if (!lhs.HasValue()) {
    return true;
  }
  return *lhs < *rhs;
}

template <typename ValueType>
bool operator>(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  if (!lhs.HasValue()) {
    return false;
  }
  if (!rhs.HasValue()) {
    return true;
  }
  return *lhs > *rhs;
}

template <typename ValueType>
bool operator<=(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  if (!lhs.HasValue()) {
    return true;
  }
  if (!rhs.HasValue()) {
    return false;
  }
  return *lhs <= *rhs;
}

template <typename ValueType>
bool operator>=(const ttv::Optional<ValueType>& lhs, const ttv::Optional<ValueType>& rhs) {
  if (!rhs.HasValue()) {
    return true;
  }
  if (!lhs.HasValue()) {
    return false;
  }
  return *lhs >= *rhs;
}

// Comparison with value
template <typename ValueType>
bool operator==(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional == value : false;
}

template <typename ValueType>
bool operator==(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional == value;
}

template <typename ValueType>
bool operator!=(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional != value : true;
}

template <typename ValueType>
bool operator!=(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional != value;
}

template <typename ValueType>
bool operator<(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional < value : true;
}

template <typename ValueType>
bool operator<(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional.HasValue() ? value < *optional : false;
}

template <typename ValueType>
bool operator>(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional > value : false;
}

template <typename ValueType>
bool operator>(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional.HasValue() ? value > *optional : true;
}

template <typename ValueType>
bool operator<=(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional <= value : true;
}

template <typename ValueType>
bool operator<=(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional.HasValue() ? value <= *optional : false;
}

template <typename ValueType>
bool operator>=(const ttv::Optional<ValueType>& optional, const ValueType& value) {
  return optional.HasValue() ? *optional >= value : false;
}

template <typename ValueType>
bool operator>=(const ValueType& value, const ttv::Optional<ValueType>& optional) {
  return optional.HasValue() ? value >= *optional : true;
}
}  // namespace ttv
