/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/eventqueue.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <atomic>

namespace ttv {
class PollingEventScheduler;
class EventQueue;
}  // namespace ttv

/**
 * PollingEventScheduler
 *
 * An implementation of IEventScheduler that is driven by calling Update().
 *
 * Update() will run all tasks with timestamp of current time or older in the current execution context.
 *
 * Shutdown() should be called before the PollingEventScheduler is destroyed.
 * Additionally, clients should wait for the shutdownTask lambda to be called back before destroying the
 * PollingEventScheduler.
 *
 * ScheduleTask() and CancelTask() should not be called after PollingEventScheduler is shut down. Otherwise, it is
 * expected that TTV_EC_NOT_INITIALIZED is returned.
 */
class ttv::PollingEventScheduler : public IEventScheduler {
 public:
  PollingEventScheduler();
  PollingEventScheduler(const std::string& name);
  virtual ~PollingEventScheduler() override;

  // IEventScheduler overrides
  virtual Result<TaskId> ScheduleTask(TaskParams&& taskParams) override;
  virtual TTV_ErrorCode CancelTask(TaskId taskId) override;
  virtual TTV_ErrorCode Shutdown(TaskFunc&& shutdownTask) override;
  virtual EventSchedulerState GetState() override;

  /**
   * Runs all tasks with timestamp of current time or older in the current execution context.
   */
  void Update();

 private:
  void CompleteShutdown();

 private:
  EventQueue mEventQueue;
  TaskFunc mShutDownTask;
  std::string mName;
  std::atomic<EventSchedulerState> mState;
};
