/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/pubsub/pubsubclientconnection.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/tracer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class TaskRunner;
class User;
class PubSubClient;
class PubSubTopicListenerHelper;

namespace pubsub {
struct ClientMessage;
struct ServerMessage;
struct ConnectionPreferenceServerMessage;
struct ConnectServerMessage;
struct DisconnectServerMessage;
struct SendMessageServerMessage;
struct SubscribeToTopicServerMessage;
struct UnsubscribeFromTopicServerMessage;
struct ShutdownServerMessage;
}  // namespace pubsub
}  // namespace ttv

// TODO: Add support for anonymous users

/**
 * Manages the pubsub connection for a user.
 *
 * Features
 *   - Topic subscription
 *   - Sending of JSON formatted messages
 *   - Automatic connection retries
 *   - Automatic keep-alive via PING/PONG messages
 *
 * Setup
 *   1) Initialize()
 *   2) Connect() and wait for stat callback indicating success.
 *
 * Listening
 *   1) Subscribe to a topic via AddTopicListener() and wait for a ITopicListener::OnTopicSubscribeStateChanged event
 *      with SubscribeState::Subscribed and TTV_EC_SUCCESS.
 *   2) Once subscribed ITopicListener::OnTopicMessageReceived will fire events as they are received for that topic.
 *
 * Unlistening
 *   1) Unsubscribe from a topic via RemoveTopicListener() and wait for ITopicListener::OnTopicSubscribeStateChanged
 * event with SubscribeState::Unsubscribed and TTV_EC_SUCCESS.
 *
 * Sending messages
 *   1) SendMessage() with the desired JSON data and an optional callback function.
 *   2) Wait for the callback function for the result.
 *
 * Disconnections
 *   When the connection to pubsub is dropped unexpecedly then all IListeners will be notified via
 * IListener::OnStateChanged. If automatic connection retries are happening then the state will be
 * TTV_CORE_PUBSUB_STATE_CONNECTING.  Otherwise, it will be TTV_CORE_PUBSUB_STATE_DISCONNECTED.
 *
 *   Registered topic listeners will be notified via ITopicListener::OnTopicSubscribeStateChanged.
 *   If a connection retry attempt is underway then the subscription state reported will be Subscribing, otherwise it
 * will be Unsubscribed.  Topic listeners are not removed from the subscription of a topic until they receive
 *   ITopicListener::OnTopicSubscribeStateChanged event with SubscribeState::Unsubscribed and TTV_EC_SUCCESS.  Any other
 * error code means that it was an issue with the connection and retries will occur automatically.
 */
class ttv::PubSubClient : public UserComponent {
 public:
  struct SubscribeState {
    enum Enum {
      Unsubscribed,
      Subscribed,
    };

    static std::string ToString(Enum state) {
      switch (state) {
        case Unsubscribed:
          return "Unsubscribed";
        case Subscribed:
          return "Subscribed";
      }

      // default fallthrough to allow compiler warning of missing enums
      return "undefined";
    }
  };

  struct ConnectionPreference {
    enum Enum {
      Manual,  //!< The client must manually call Connect() and Disconnect().
      OnDemand  //!< Connects automatically if there are topic listeners registered and disconnects when all have been unregistered.
    };
  };

  class IListener {
   public:
    /**
     * Indicates the current connection state of pubsub.  If disconnected for any reason then topic subscriptions will
     * be restored when the connection is reestablished.
     */
    virtual void OnStateChanged(PubSubClient* source, PubSubState state, TTV_ErrorCode ec) = 0;
    virtual ~IListener() = default;
  };

  class ITopicListener {
   public:
    virtual ~ITopicListener() = default;

    /**
     * Indicates changes to the topic subscription state on the socket.  After receiving SubscribeState::Subscribed the
     * client should resync all absolute state in order for deltas to properly apply to the local data.
     */
    virtual void OnTopicSubscribeStateChanged(
      PubSubClient* source, const std::string& topic, SubscribeState::Enum state, TTV_ErrorCode ec) = 0;
    /**
     * Received a message on a topic.
     */
    virtual void OnTopicMessageReceived(PubSubClient* source, const std::string& topic, const json::Value& msg) = 0;
    /**
     * Indicates the the listener is no longer going to receive topic events and the reference to the listener is
     * removed.  This is independent of topic subscription and there could be other registered topic listeners for the
     * same topic.
     */
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) = 0;
  };

  using SendMessageCallback = std::function<void(PubSubClient* source, TTV_ErrorCode ec)>;

 public:
  PubSubClient(std::shared_ptr<User> user, std::shared_ptr<SettingRepository> settingRepository);
  virtual ~PubSubClient() override;

  PubSubState GetConnectionState() const { return mConnectionState.client; }

  // Subscription for the connection state of the PubSub component.
  void AddListener(std::shared_ptr<IListener> listener);
  void RemoveListener(std::shared_ptr<IListener> listener);

  TTV_ErrorCode Initialize() override;
  virtual TTV_ErrorCode Connect();
  TTV_ErrorCode Disconnect();

  TTV_ErrorCode SetConnectionPreference(ConnectionPreference::Enum preference);

  TTV_ErrorCode SendMessage(const json::Value& jMessage, SendMessageCallback callback);

  // Subscription for subscriptions about topic events
  TTV_ErrorCode AddTopicListener(const std::string& topic, std::shared_ptr<ITopicListener> listener);
  TTV_ErrorCode RemoveTopicListener(const std::string& topic, std::shared_ptr<ITopicListener> listener);
  std::vector<std::string> GetSubscribedTopics() const;

  // Component overrides
  TTV_ErrorCode Shutdown() override;
  virtual void Update() override;

  // UserComponent overrides
  virtual std::string GetLoggerName() const override;

  static std::string GetComponentName() { return "ttv::PubSubClient"; }

 private:
  struct Topic {
    Topic();

    std::string topic;
    EventSource<ITopicListener> listeners;
  };

  struct TopicState {
    SubscribeState::Enum state;  //!< The current state of the connection.
  };

  class ConnectionListener : public PubSubClientConnection::IListener {
   public:
    ConnectionListener(PubSubClient* pubsub);

    // PubSubClientConnection::IListener Implementation
    virtual void OnReconnectReceived(PubSubClientConnection* connection) override;
    virtual void OnConnectionStateChanged(
      PubSubClientConnection* connection, PubSubState state, TTV_ErrorCode ec) override;
    virtual void OnTopicSubscriptionChanged(PubSubClientConnection* connection, const std::string& topic,
      PubSubClientConnection::TopicSubscriptionState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClientConnection* connection, const std::string& topic, const json::Value& message) override;
    virtual void OnPongTimeout(PubSubClientConnection* connection) override;
    virtual void OnAuthenticationError(PubSubClientConnection* connection, TTV_ErrorCode ec,
      const std::shared_ptr<const OAuthToken>& authToken) override;

   private:
    PubSubClient* mPubSub;
  };

  // PubSubClientConnection::IListener Implementation
  void OnReconnectReceived(PubSubClientConnection* connection);
  void OnConnectionStateChanged(PubSubClientConnection* connection, PubSubState state, TTV_ErrorCode ec);
  void OnTopicSubscriptionChanged(PubSubClientConnection* connection, const std::string& topic,
    PubSubClientConnection::TopicSubscriptionState::Enum state, TTV_ErrorCode ec);
  void OnTopicMessageReceived(PubSubClientConnection* connection, const std::string& topic, const json::Value& message);
  void OnPongTimeout(PubSubClientConnection* connection);
  void OnAuthenticationEror(
    PubSubClientConnection* connection, TTV_ErrorCode ec, const std::shared_ptr<const OAuthToken>& authToken);

  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  bool AnyConnected() const;
  bool MainConnected() const;
  void ThreadProc();
  void SetConnectionState(PubSubState state, TTV_ErrorCode ec);

  void UpdateTopicSubscription(const std::string& topic);
  void PerformReconnect();
  TTV_ErrorCode SyncTopicSubscriptions();
  SubscribeState::Enum GetEffectiveTopicState(const std::string& topic);

  TTV_ErrorCode ProcessRequestQueue();
  TTV_ErrorCode ProcessConnectionPreference(std::shared_ptr<pubsub::ConnectionPreferenceServerMessage> msg);
  TTV_ErrorCode ProcessConnect(std::shared_ptr<pubsub::ConnectServerMessage> msg);
  TTV_ErrorCode ProcessDisconnect(std::shared_ptr<pubsub::DisconnectServerMessage> msg);
  TTV_ErrorCode ProcessSendMessage(std::shared_ptr<pubsub::SendMessageServerMessage> msg);
  TTV_ErrorCode ProcessSubscribeToTopic(std::shared_ptr<pubsub::SubscribeToTopicServerMessage> msg);
  TTV_ErrorCode ProcessUnsubscribeFromTopic(std::shared_ptr<pubsub::UnsubscribeFromTopicServerMessage> msg);
  TTV_ErrorCode ProcessShutdown(std::shared_ptr<pubsub::ShutdownServerMessage> msg);

  TTV_ErrorCode PerformDisconnect();
  TTV_ErrorCode ScheduleConnect(TTV_ErrorCode ec);
  TTV_ErrorCode AttemptConnection();

  ConcurrentQueue<std::shared_ptr<pubsub::ClientMessage>> mIncomingQueue;
  ConcurrentQueue<std::shared_ptr<pubsub::ServerMessage>> mOutgoingQueue;

  std::map<std::string, std::shared_ptr<Topic>> mTopics;

  std::shared_ptr<IThread> mThread;
  std::shared_ptr<ConnectionListener> mConnectionListener;
  std::shared_ptr<PubSubClientConnection> mConnection;  //!< The active connection.
  std::shared_ptr<PubSubClientConnection>
    mDyingConnection;  //!< The connection that might be alive for a short time after a RECONNECT is received from the server.
  std::shared_ptr<SettingRepository> mSettingRepository;
  EventSource<IListener> mListeners;
  json::FastWriter mJsonWriter;
  RetryTimer mConnectionRetryTimer;
  ClientServerValue<PubSubState> mConnectionState;
  ConnectionPreference::Enum mConnectionPreference;
  bool mConnectionDesired;  //!< Whether or not the client has called Connect().
};

/**
 * A helper listener proxy which makes it easier to manage subscription state.
 */
class ttv::PubSubTopicListenerHelper : public PubSubClient::ITopicListener,
                                       public std::enable_shared_from_this<PubSubTopicListenerHelper> {
 public:
  enum class State {
    Initialized,
    ShuttingDown,
    Shutdown,
  };

 public:
  PubSubTopicListenerHelper(
    std::shared_ptr<PubSubClient> pubsub, std::shared_ptr<PubSubClient::ITopicListener> listener);
  virtual ~PubSubTopicListenerHelper() override;

  State GetState() const { return mState; }
  PubSubClient::SubscribeState::Enum GetSubscriptionState(const std::string& topic);

  TTV_ErrorCode Subscribe(const std::string& topic);
  TTV_ErrorCode Unsubscribe(const std::string& topic);
  bool ContainsTopic(const std::string& topic) const;

  void Shutdown();

  // PubSubClient::ITopicListener implementation
  virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
    PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
  virtual void OnTopicMessageReceived(PubSubClient* source, const std::string& topic, const json::Value& msg) override;
  virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

 protected:
  struct Entry {
    PubSubClient::SubscribeState::Enum state;
    bool subscriptionDesired;
  };

  std::map<std::string, Entry> mSubscriptionStates;
  std::shared_ptr<PubSubClient> mPubSub;
  std::weak_ptr<PubSubClient::ITopicListener> mListener;
  State mState;
};
