/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class OAuthToken;
class PubSubClientConnection;
class User;

namespace pubsub {
struct ClientMessage;
struct ServerMessage;
struct ConnectServerMessage;
struct DisconnectServerMessage;
struct SendMessageServerMessage;
struct SubscribeToTopicServerMessage;
struct UnsubscribeFromTopicServerMessage;

constexpr char kPubSubEndpointUriKey[] = "kPubSubEndpointUriKey";
}  // namespace pubsub
}  // namespace ttv

/**
 * A single use connection used by PubSubClient.  Once it has made a connection attempt it should not be used again once
 * disconnected. This is a single threaded component.
 */
class ttv::PubSubClientConnection {
 public:
  struct TopicSubscriptionState {
    enum Enum { Unsubscribed, Subscribing, Subscribed, Unsubscribing };
  };

  class IListener {
   public:
    virtual ~IListener() = default;
    virtual void OnReconnectReceived(PubSubClientConnection* connection) = 0;
    virtual void OnConnectionStateChanged(PubSubClientConnection* connection, PubSubState state, TTV_ErrorCode ec) = 0;
    virtual void OnTopicSubscriptionChanged(PubSubClientConnection* connection, const std::string& topic,
      TopicSubscriptionState::Enum state, TTV_ErrorCode ec) = 0;
    virtual void OnTopicMessageReceived(
      PubSubClientConnection* connection, const std::string& topic, const json::Value& message) = 0;
    virtual void OnPongTimeout(PubSubClientConnection* connection) = 0;
    virtual void OnAuthenticationError(
      PubSubClientConnection* connection, TTV_ErrorCode ec, const std::shared_ptr<const OAuthToken>& oauthToken) = 0;
  };

  using InternalCallback = std::function<void(TTV_ErrorCode ec, const std::string& error, const json::Value& result)>;

 public:
  PubSubClientConnection(std::shared_ptr<User> user, SettingRepository* settingsRepository);

  PubSubState GetState() const { return mState; }
  TopicSubscriptionState::Enum GetTopicState(const std::string& topic) const;

  void AddListener(const std::shared_ptr<IListener>& listener);
  void RemoveListener(const std::shared_ptr<IListener>& listener);

  TTV_ErrorCode Connect();
  TTV_ErrorCode Disconnect();
  TTV_ErrorCode Send(const std::string& text);
  bool Connected();

  TTV_ErrorCode Listen(const std::string& topic, std::shared_ptr<User> user);
  TTV_ErrorCode Unlisten(const std::string& topic);
  TTV_ErrorCode SendMessageOverSocket(const json::Value& root);
  TTV_ErrorCode SendNullDataMessage(const std::string& type);

  TTV_ErrorCode PollSocket();

  void Update();

  uint32_t GetConnectionIndex() const { return mConnectionIndex; }

 private:
  struct OutstandingRequest {
    OutstandingRequest();

    std::string nonce;
    InternalCallback callback;
    Timestamp requestTime;  //!< Used to timeout requests.
  };

  TTV_ErrorCode HandleIncomingMessage();
  TTV_ErrorCode InitiatePing();
  void SetConnectionState(PubSubState state, TTV_ErrorCode ec);
  void Log(MessageLevel level, const char* format, ...);

  std::weak_ptr<User> mUser;
  std::shared_ptr<IWebSocket> mSocket;
  std::string mReadBuffer;
  std::map<std::string, TopicSubscriptionState::Enum> mTopicStates;
  std::map<std::string, OutstandingRequest> mOutstandingRequests;
  json::Reader mJsonReader;
  json::FastWriter mJsonWriter;
  WaitForExpiry mPingTimer;
  WaitForExpiry mPongTimer;
  PubSubState mState;
  EventSource<IListener> mListeners;
  uint32_t mConnectionIndex;
};
