/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class OAuthToken;

namespace pubsub {
struct ClientMessageType {
  enum Enum {
    SendMessageResult,
    StateChanged,
    MessageReceived,
    TopicSubscriptionChanged,
    TopicListenerRemoved,
    AuthErrorReceived,
    ShutdownComplete
  };
};

struct ClientMessage {
  ClientMessageType::Enum type;
  TTV_ErrorCode ec;
};

struct SendMessageResultClientMessage : public ClientMessage {
  SendMessageResultClientMessage(TTV_ErrorCode ec, PubSubClient::SendMessageCallback callback);

  PubSubClient::SendMessageCallback callback;
};

struct StateChangedClientMessage : public ClientMessage {
  StateChangedClientMessage(PubSubState state, TTV_ErrorCode ec);

  std::vector<std::shared_ptr<PubSubClient::IListener>> listeners;
  PubSubState state;
};

struct MessageReceivedClientMessage : public ClientMessage {
  MessageReceivedClientMessage(const std::string& topic, const json::Value& data);

  std::vector<std::shared_ptr<PubSubClient::ITopicListener>> listeners;
  std::string topic;
  json::Value data;
};

struct TopicSubscriptionChangedClientMessage : public ClientMessage {
  TopicSubscriptionChangedClientMessage(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec);

  std::vector<std::shared_ptr<PubSubClient::ITopicListener>> listeners;
  std::string topic;
  PubSubClient::SubscribeState::Enum state;
};

struct TopicListenerRemovedClientMessage : public ClientMessage {
  TopicListenerRemovedClientMessage(
    std::shared_ptr<PubSubClient::ITopicListener> listener, const std::string& topic, TTV_ErrorCode ec);

  std::string topic;
  std::shared_ptr<PubSubClient::ITopicListener> listener;
};

struct AuthErrorReceivedClientMessage : public ClientMessage {
  AuthErrorReceivedClientMessage(TTV_ErrorCode ec, const std::shared_ptr<const OAuthToken>& authToken);

  std::shared_ptr<const OAuthToken> authToken;
};

struct ShutdownCompleteClientMessage : public ClientMessage {
  ShutdownCompleteClientMessage();
};

struct ServerMessageType {
  enum Enum {
    Connect,
    Disconnect,
    SendMessage,
    SubscribeToTopic,
    UnsubscribeFromTopic,
    ConnectionPreference,
    Shutdown
  };
};

struct ServerMessage {
  ServerMessageType::Enum type;
};

struct ConnectionPreferenceServerMessage : public ServerMessage {
  ConnectionPreferenceServerMessage(PubSubClient::ConnectionPreference::Enum preference);

  PubSubClient::ConnectionPreference::Enum preference;
};

struct ConnectServerMessage : public ServerMessage {
  ConnectServerMessage();
};

struct DisconnectServerMessage : public ServerMessage {
  DisconnectServerMessage();
};

struct SendMessageServerMessage : public ServerMessage {
  SendMessageServerMessage(const json::Value& data, PubSubClient::SendMessageCallback callback);

  json::Value data;
  PubSubClient::SendMessageCallback callback;
};

struct SubscribeToTopicServerMessage : public ServerMessage {
  SubscribeToTopicServerMessage(const std::string& topic, std::shared_ptr<const OAuthToken> oauth,
    std::shared_ptr<PubSubClient::ITopicListener> listener);

  std::string topic;
  std::shared_ptr<const OAuthToken> oauth;
  std::shared_ptr<PubSubClient::ITopicListener> listener;
};

struct UnsubscribeFromTopicServerMessage : public ServerMessage {
  UnsubscribeFromTopicServerMessage(const std::string& topic, std::shared_ptr<PubSubClient::ITopicListener> listener);

  std::string topic;
  std::shared_ptr<PubSubClient::ITopicListener> listener;
};

struct ShutdownServerMessage : public ServerMessage {
  ShutdownServerMessage();
};
}  // namespace pubsub
}  // namespace ttv
