/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/user/userrepository.h"

#include <memory>
#include <vector>

namespace ttv {
class PubSubClient;
class PubSubComponentBase;
template <typename LISTENER>
class PubSubComponent;
}  // namespace ttv

class ttv::PubSubComponentBase : public ttv::UserComponent {
 public:
  using DisposerFunc = std::function<void()>;

 public:
  PubSubComponentBase(const std::shared_ptr<User>& user);

  void SetDisposer(DisposerFunc&& func) { mDisposerFunc = func; }

  TTV_ErrorCode Dispose();

  // UserComponent overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override = 0;

  virtual void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) = 0;
  virtual void OnTopicMessageReceived(const std::string& topic, const json::Value& msg) = 0;
  virtual void OnTopicListenerRemoved(const std::string& topic, TTV_ErrorCode ec);

 protected:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  virtual void AddTopic(const std::string& topic);
  virtual TTV_ErrorCode SubscribeTopics();

  class PubSubTopicListener : public PubSubClient::ITopicListener {
   public:
    PubSubTopicListener(PubSubComponentBase* owner) : mOwner(owner) {}

    // PubSubClient::ITopicListener implementation
    virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
      PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClient* source, const std::string& topic, const json::Value& msg) override;
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

   private:
    PubSubComponentBase* mOwner;
  };

  std::shared_ptr<PubSubClient> mPubSub;
  std::shared_ptr<PubSubTopicListener> mPubSubTopicListener;
  std::shared_ptr<PubSubTopicListenerHelper> mPubSubTopicListenerHelper;
  DisposerFunc mDisposerFunc;
  std::vector<std::string> mPubSubTopics;
};

/**
 * Manages listeners for pub-sub updates on a channel
 */
template <typename LISTENER>
class ttv::PubSubComponent : public ttv::PubSubComponentBase {
 public:
  PubSubComponent(const std::shared_ptr<User>& user) : PubSubComponentBase(user) {}

  virtual void CompleteShutdown() override {
    PubSubComponentBase::CompleteShutdown();
    mListener.reset();
  }

  // Call this setter before Initialize()
  void SetListener(const std::shared_ptr<LISTENER>& listener) { mListener = listener; }

 protected:
  std::shared_ptr<LISTENER> mListener;
};
