/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <random>

namespace ttv {
namespace random {
/**
 * Provide an interface compatible with STL random number generator
 * Can be redefined for each platform where default_random_engine is sub-optimal.
 */
class IGenerator {
 public:
  virtual ~IGenerator() = default;

  /**
   * The type of random number generation
   */
  typedef uint64_t result_type;

  /**
   * Generates a random number
   */
  virtual uint64_t operator()() = 0;

  /**
   * Returns the minimum value potentially returned by operator()
   */
  static constexpr result_type(min)() {
    // return minimum possible generated value
    return std::numeric_limits<result_type>::min();
  }

  /**
   * Returns the maximum value potentially returned by operator()
   */
  static constexpr result_type(max)() {
    // return maximum possible generated value
    return std::numeric_limits<result_type>::max();
  }
};

/**
 * Return a reference to a thread-safe Generator interface.
 */
IGenerator& GetGenerator();

}  // namespace random
}  // namespace ttv
