/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <functional>
#include <memory>

namespace ttv {
template <typename RESOURCE, typename FACTORY>
class ResourceFactoryChain {
 public:
  using CreateFunc =
    std::function<TTV_ErrorCode(const std::shared_ptr<FACTORY>& factory, std::shared_ptr<RESOURCE>& result)>;
  using QueryFunc = std::function<bool(const std::shared_ptr<FACTORY>& factory)>;

 public:
  ResourceFactoryChain(const std::string& name) : mName(name), mRegisteredDefaults(false) {}

  virtual ~ResourceFactoryChain() {}

  virtual TTV_ErrorCode Register(const std::shared_ptr<FACTORY>& factory) {
    if (factory == nullptr) {
      ttv::trace::Message("Core", MessageLevel::Error, "ResourceFactoryChain::Register: null factory given");

      return TTV_EC_INVALID_ARG;
    }

    auto iter = std::find(mChain.begin(), mChain.end(), factory);
    if (iter != mChain.end()) {
      ttv::trace::Message("Core", MessageLevel::Error, "ResourceFactoryChain::Register: factory already registered");

      return TTV_EC_INVALID_ARG;
    }

    mChain.insert(mChain.begin(), factory);

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode Unregister(const std::shared_ptr<FACTORY>& factory) {
    if (factory == nullptr) {
      ttv::trace::Message("Core", MessageLevel::Error, "ResourceFactoryChain::Unregister: null factory given");

      return TTV_EC_INVALID_ARG;
    }

    auto iter = std::find(mChain.begin(), mChain.end(), factory);
    if (iter == mChain.end()) {
      ttv::trace::Message("Core", MessageLevel::Error, "ResourceFactoryChain::Register: factory not registered");

      return TTV_EC_INVALID_ARG;
    }

    mChain.erase(iter);

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode Create(const CreateFunc& createFunc, std::shared_ptr<RESOURCE>& result) const {
    result.reset();

    if (mChain.empty()) {
      ttv::trace::Message("Core", MessageLevel::Error, "There are no %s factories registered", mName.c_str());
      return TTV_EC_NO_FACTORIES_REGISTERED;
    }

    auto copy = mChain;

    for (const auto& factory : copy) {
      TTV_ErrorCode ec = createFunc(factory, result);
      if (TTV_SUCCEEDED(ec) && result != nullptr) {
        break;
      }
    }

    if (result != nullptr) {
      return TTV_EC_SUCCESS;
    } else {
      return TTV_EC_UNIMPLEMENTED;
    }
  }

  bool BoolQuery(const QueryFunc& queryFunc) const {
    auto copy = mChain;

    for (const auto& factory : copy) {
      if (queryFunc(factory)) {
        return true;
      }
    }

    return false;
  }

  void SetRegisteredDefaults() { mRegisteredDefaults = true; }

  bool RegisteredDefaults() const { return mRegisteredDefaults; }

  bool Empty() const { return mChain.empty(); }

 protected:
  std::string mName;
  std::vector<std::shared_ptr<FACTORY>> mChain;
  bool mRegisteredDefaults;
};
}  // namespace ttv
