/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/variant.h"

namespace ttv {
/**
 * Templated Result class that can acts as a union between TTV_ErrorCode and a Result (of any type).
 *
 * Results can be of two main types:
 *   ErrorResult: Holds a TTV_ErrorCode
 *   SuccessResult Holds a result of ResultType
 *
 * IsSuccess()/IsError() can be used to deduce the type of the Result, and GetErrorCode()/GetResult() is used to retrive
 * the Error Code or the Result. GetResult() returns the result and should only be called if it is a SuccessResult (i.e.
 * IsSuccess() == true). GetErrorCode() returns the error code
 *
 * MakeSuccessResult and MakeErrorResult should be used to create Result objects.
 *
 * Results can be freely copy/move constructed/assigned to each other if the ResultTypes are convertible to each other.
 */
template <typename ResultType>
class Result;

/**
 * Partial specialization of Result for type void.
 * This is only used for ErrorResults, i.e. Results that only hold errors.
 * Result<void> can be easily converted into Result<ResultTypeParam> through the Result<void> constructor in
 * Result<ResultTypeParam>.
 */
template <>
class Result<void>;

/**
 * Makes a Result<ResultTypeParam> object with a result of ResultTypeParam.
 * GetResult() can be used to retrive the result.
 */
template <typename ResultTypeParam>
Result<std::decay_t<ResultTypeParam>> MakeSuccessResult(ResultTypeParam&&);

/**
 * Makes a Result<ResultTypeParam> object with a result of ResultTypeParam and an TTV_ErrorCode.
 * The TTV_ErrorCode must be a Warning TTV_ErrorCode that still implies success.
 *
 * GetResult() can be used to retrive the result.
 * IsWarning() can be used to retrive the Warning TTV_ErrorCode.
 */
template <typename ResultTypeParam>
Result<std::decay_t<ResultTypeParam>> MakeWarningResult(TTV_ErrorCode, ResultTypeParam&&);

/**
 * Makes a Result<void> object with a TTV_ErrorCode.
 * The TTV_ErrorCode must be a Error TTV_ErrorCode and imply failure.
 *
 * GetErrorCode() can be used to retrive the TTV_ErrorCode.
 * Calling GetResult() on this will assert and fail.
 */
ttv::Result<void> MakeErrorResult(TTV_ErrorCode ec);
}  // namespace ttv

template <>
class ttv::Result<void> {
 public:
  /**
   * Result<void> is only used for ErrorResults.
   */
  bool IsSuccess() const { return false; }

  bool IsError() const { return true; }

  TTV_ErrorCode GetErrorCode() const { return mError; }

 private:
  /**
   * Private constructor, as Error results should only be constructed with MakeErrorResult.
   * Result<void> does not take in a result, and only a TTV_ErrorCode.
   */
  explicit Result(TTV_ErrorCode ec) : mError(ec) { TTV_ASSERT(TTV_FAILED(ec)); }

  TTV_ErrorCode mError;

  friend ttv::Result<void> MakeErrorResult(TTV_ErrorCode ec);
};

template <typename ResultType>
class ttv::Result {
 public:
  bool IsSuccess() const { return mVariant.template Is<ResultType>(); }

  bool IsError() const { return mVariant.template Is<ErrorContainer>(); }

  TTV_ErrorCode GetErrorCode() const {
    return mVariant.template Is<ErrorContainer>() ? mVariant.template As<ErrorContainer>().code : TTV_EC_SUCCESS;
  }

  /**
   * Should only be called if Result is a SuccessResult (IsSuccess() == true / IsError() == false).
   */
  ResultType& GetResult() {
    TTV_ASSERT(mVariant.template Is<ResultType>());
    return mVariant.template As<ResultType>();
  }

  const ResultType& GetResult() const {
    TTV_ASSERT(mVariant.template Is<ResultType>());
    return mVariant.template As<ResultType>();
  }

  /**
   * Constructor to convert a Result<void> with no type to a Result<ResultParam> with a result type.
   */
  Result(const Result<void>& result) : mVariant(ErrorContainer{result.GetErrorCode()}) {}

  Result& operator=(const Result<void>& result) {
    mVariant = ErrorContainer{result.GetErrorCode()};
    return *this;
  }

  template <typename SourceResultType>
  using CanConvertFrom = std::enable_if_t<std::is_convertible<SourceResultType, ResultType>::value>;

  /**
   * Copy from another result with any convertible result type.
   */
  template <typename SourceResultType, typename = CanConvertFrom<SourceResultType>>
  Result(const Result<SourceResultType>& src) : mVariant(ConvertSourceResultToVariant(src)) {}

  /**
   * Move from another result with any convertible result type.
   */
  template <typename SourceResultType, typename = CanConvertFrom<SourceResultType>>
  Result(Result<SourceResultType>&& src) : mVariant(ConvertSourceResultToVariant(std::move(src))) {}

  /**
   * Copy assignment from another result with any convertible result type.
   */
  template <typename SourceResultType, typename = CanConvertFrom<SourceResultType>>
  Result& operator=(const Result<SourceResultType>& src) {
    mVariant = ConvertSourceResultToVariant(src);
    return *this;
  }

  /**
   * Move assignment from another result with any convertible result type.
   */
  template <typename SourceResultType, typename = CanConvertFrom<SourceResultType>>
  Result& operator=(Result<SourceResultType>&& src) {
    mVariant = ConvertSourceResultToVariant(std::move(src));
    return *this;
  }

 private:
  /**
   * Private constructor, as Success and Warning results should only be constructed with
   * MakeSuccessResult/MakeWarningResult.
   */
  template <typename ResultTypeParam,
    typename = std::enable_if_t<std::is_same<std::decay_t<ResultTypeParam>, ResultType>::value>>
  explicit Result(ResultTypeParam&& result) : mVariant(std::forward<ResultTypeParam>(result)) {}

 private:
  /**
   * This class is here just to make sure we have a distinct type for our error case, rather
   * than just a uint32_t which could collide with a number of different result types.
   */
  struct ErrorContainer {
    ErrorContainer(TTV_ErrorCode ec) : code(ec) { TTV_ASSERT(!TTV_SUCCEEDED(ec)); }

    TTV_ErrorCode code;
  };

  template <typename SourceResultType>
  static ttv::Variant<ErrorContainer, ResultType> ConvertSourceResultToVariant(Result<SourceResultType>&& result) {
    if (result.IsSuccess()) {
      return static_cast<ResultType>(std::move(result.GetResult()));
    } else {
      return ErrorContainer{result.GetErrorCode()};
    }
  }

  template <typename SourceResultType>
  static ttv::Variant<ErrorContainer, ResultType> ConvertSourceResultToVariant(const Result<SourceResultType>& result) {
    if (result.IsSuccess()) {
      return static_cast<ResultType>(result.GetResult());
    } else {
      return ErrorContainer{result.GetErrorCode()};
    }
  }

  ttv::Variant<ErrorContainer, ResultType> mVariant;

  template <typename ResultTypeParam>
  friend ttv::Result<std::decay_t<ResultTypeParam>> ttv::MakeSuccessResult(ResultTypeParam&& result);
};

template <typename ResultTypeParam>
ttv::Result<std::decay_t<ResultTypeParam>> ttv::MakeSuccessResult(ResultTypeParam&& result) {
  return ttv::Result<std::decay_t<ResultTypeParam>>(std::forward<ResultTypeParam>(result));
}

inline ttv::Result<void> ttv::MakeErrorResult(TTV_ErrorCode ec) {
  return ttv::Result<void>(ec);
}
