/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

/**
 * @file
 *
 * Standard protocol names:
 *   - tcp, raw: An insecure TCP connection
 *   - udp: An insecure UDP
 *   - tls: A secure tcp connection
 *   - ws: An insecure websocket connection
 *   - wss: A secure websocket connection
 *   - file: A file reader and writer
 */

#pragma once

#include "twitchsdk/core/sockettracker.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <array>
#include <memory>

#define SOCKET_SUCCEEDED(ec) (TTV_SUCCEEDED(ec) || (ec == TTV_EC_SOCKET_EWOULDBLOCK))
#define SOCKET_FAILED(ec) (TTV_FAILED(ec) && (ec != TTV_EC_SOCKET_EWOULDBLOCK))

namespace ttv {
class ISocket;
class IWebSocket;
class ISocketFactory;
class IWebSocketFactory;
class BufferedSocket;

/**
 * Initializes the socket library subsystem.  This must be done once before any socket factories are registered.
 *
 * @see ShutdownSocketLibrary
 *
 * @return
 *   - TTV_EC_SUCCESS: There is at least one factory registered that supports the given protocol.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_ALREADY_INITIALIZED: The socket library subsystem has already been initialized.
 */
TTV_ErrorCode InitializeSocketLibrary();

/**
 * Shuts down the socket library subsystem.  This should only be done after all modules are shut down
 * and no more clients are using sockets.  It is intended to be called some time after InitializeSocketLibrary()
 * is called but it is safe to call ShutdownSocketLibrary() even if the library is already shut down
 * or was never initialized.
 *
 * @see InitializeSocketLibrary
 *
 * @return
 *   - TTV_EC_SUCCESS: There is at least one factory registered that supports the given protocol.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 */
TTV_ErrorCode ShutdownSocketLibrary();

/**
 * Registers an ISocketFactory to be used when creating ISocket instances.
 *
 * @see CreateSocket
 * @see UnregisterSocketFactory
 * @param[in] factory The factory to register.  Cannot be null or already registered.
 * @return
 *   - TTV_EC_SUCCESS: The factory was successfully registered.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_INVALID_ARG: The factory is null or was already registered.
 */
TTV_ErrorCode RegisterSocketFactory(const std::shared_ptr<ISocketFactory>& factory);

/**
 * Unregisters an ISocketFactory previously registered via RegisterSocketFactory().
 *
 * @see RegisterSocketFactory
 * @param[in] factory The factory to unregister.  Cannot be null and must be previously registered.
 * @return
 *   - TTV_EC_SUCCESS: The factory was successfully unregistered.
 *   - TTV_EC_INVALID_ARG: The factory is null or was not previously registered.
 */
TTV_ErrorCode UnregisterSocketFactory(const std::shared_ptr<ISocketFactory>& factory);

/**
 * Registers an IWebSocketFactory to be used when creating IWebSocket instances.
 *
 * @see CreateWebSocket
 * @see UnregisterWebSocketFactory
 * @param[in] factory The factory to register.  Cannot be null or already registered.
 * @return
 *   - TTV_EC_SUCCESS: The factory was successfully registered.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_INVALID_ARG: The factory is null or was already registered.
 */
TTV_ErrorCode RegisterWebSocketFactory(const std::shared_ptr<IWebSocketFactory>& factory);

/**
 * Unregisters an IWebSocketFactory previously registered via RegisterWebSocketFactory().
 *
 * @see RegisterWebSocketFactory
 * @param[in] factory The factory to unregister.  Cannot be null and must be previously registered.
 * @return
 *   - TTV_EC_SUCCESS: The factory was successfully unregistered.
 *   - TTV_EC_INVALID_ARG: The factory is null or was not previously registered.
 */
TTV_ErrorCode UnregisterWebSocketFactory(const std::shared_ptr<IWebSocketFactory>& factory);

/**
 * Determines if there is a registered ISocketFactory that supports the named protocol.
 *
 * @see RegisterSocketFactory
 * @param[in] protocol The protocol to check.  It is the short protocol name, such as "http".
 * @return
 *   - TTV_EC_SUCCESS: There is at least one factory registered that supports the given protocol.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_NO_FACTORIES_REGISTERED: There are no registed factories.
 *   - TTV_EC_UNIMPLEMENTED: There are factories registered but none of them support the protocol.
 */
TTV_ErrorCode IsSocketProtocolSupported(const std::string& protocol);

/**
 * Determines if there is a registered IWebSocketFactory that supports the named protocol.
 *
 * @see RegisterWebSocketFactory
 * @param[in] protocol The protocol to check.  It is the short protocol name, such as "ws".
 * @return
 *   - TTV_EC_SUCCESS: There is at least one factory registered that supports the given protocol.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_NO_FACTORIES_REGISTERED: There are no registed factories.
 *   - TTV_EC_UNIMPLEMENTED: There are factories registered but none of them support the protocol.
 */
TTV_ErrorCode IsWebSocketProtocolSupported(const std::string& protocol);

/**
 * Attempts to create an ISocket instance that supports the given URI.
 *
 * When creating a new ISocket the registered factories will be visited in order of newest to oldest
 * registration until one is found that supports the requested protocol.
 *
 * @see RegisterSocketFactory
 * @param[in] uri The full uri to create the socket for.
 * @param[out] result The resulting ISocket instance if it was able to be created.
 * @return
 *   - TTV_EC_SUCCESS: The result has been populated with an ISocket instance to use.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_NO_FACTORIES_REGISTERED: There are no registed factories.
 *   - TTV_EC_UNIMPLEMENTED: There are factories registered but none of them support the protocol.
 */
TTV_ErrorCode CreateSocket(const std::string& uri, std::shared_ptr<ISocket>& result);

/**
 * Attempts to create an IWebSocket instance that supports the given URI.
 *
 * When creating a new IWebSocket the registered factories will be visited in order of newest to oldest
 * registration until one is found that supports the requested protocol.
 *
 * @see RegisterWebSocketFactory
 * @param[in] uri The full uri to create the socket for.
 * @param[out] result The resulting IWebSocket instance if it was able to be created.
 * @return
 *   - TTV_EC_SUCCESS: The result has been populated with an IWebSocket instance to use.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_NO_FACTORIES_REGISTERED: There are no registed factories.
 *   - TTV_EC_UNIMPLEMENTED: There are factories registered but none of them support the protocol.
 */
TTV_ErrorCode CreateWebSocket(const std::string& uri, std::shared_ptr<IWebSocket>& result);
}  // namespace ttv

/**
 * This interface provides generic, synchronous access to a socket.  It can wrap any low-level bidirectional byte
 * stream.  An instance of ISocket is generally bound to a single endpoint.  The connection can be opened and closed
 * as many times as desired but it can't be changed to point to another peer.
 */
class ttv::ISocket {
 public:
  /**
   * Destructor.
   */
  virtual ~ISocket() = default;

  /**
   * Synchronously connects to the endpoint.
   *
   * @see Disconnect
   * @return
   *   - TTV_EC_SUCCESS: The socket has successfully connected and is ready to exchange data.
   *   - TTV_EC_SOCKET_EALREADY: Already connected.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Connect() = 0;

  /**
   * Synchronously disconnects from the endpoint.  It is safe to call Disconnect() if not connected.
   *
   * @see Connect
   * @return
   *   - TTV_EC_SUCCESS: The socket has successfully disconnected.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Disconnect() = 0;

  /**
   * Attempts to send some bytes over the socket.  This flavor of Send() is a helper which guarantees that
   * all bytes are sent or where is an error on the socket.
   *
   * @param[in] buffer The buffer to store the received bytes in.
   * @param[in] length The number of bytes to send.
   * @return
   *   - TTV_EC_SUCCESS: The bytes have been sent.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Send(const uint8_t* buffer, size_t length);

  /**
   * Attempts to send some bytes over the socket.  It's possible that the socket is unable to send able
   * to send any more bytes right now but will be in the near future.  Thus, the number actually sent
   * may be less than the size of the passed in buffer.
   *
   * @param[in] buffer The buffer to store the received bytes in.
   * @param[in] length The number of bytes to send.
   * @param[out] sent The number of bytes actually sent.
   * @return
   *   - TTV_EC_SUCCESS: Some bytes have been sent.
   *   - TTV_EC_SOCKET_EWOULDBLOCK: The socket is connected but wasn't able to send any bytes right now.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Send(const uint8_t* buffer, size_t length, size_t& sent) = 0;

  /**
   * Attempts to receive some bytes from the socket.
   *
   * @param[in] buffer The buffer to store the received bytes in.
   * @param[in] length The largest number of bytes to receive.
   * @param[out] received The number of bytes actually received.
   * @return
   *   - TTV_EC_SUCCESS: Some bytes have been received.
   *   - TTV_EC_SOCKET_EWOULDBLOCK: The socket is connected but there aren't any bytes available to be read.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Recv(uint8_t* buffer, size_t length, size_t& received) = 0;

  /**
   * Returns the total number of bytes that have been sent over the socket during the current connection.
   *
   * @return The number of bytes sent during the current connection.
   */
  virtual uint64_t TotalSent() = 0;

  /**
   * Returns the total number of bytes that have been received from the socket during the current connection.
   *
   * @return The number of bytes received during the current connection.
   */
  virtual uint64_t TotalReceived() = 0;

  /**
   * Determines if the socket is currently connected.
   *
   * @return Whether or not currently connected.
   */
  virtual bool Connected() = 0;
};

/**
 * This interface provides generic, synchronous access to a web socket (`ws` protocol).  An instance of IWebSocket is
 * generally bound to a single endpoint.  The connection can be opened and closed as many times as desired but it can't
 * be changed to point to another peer.
 */
class ttv::IWebSocket {
 public:
  /**
   * The type of data in a frame.
   */
  enum class MessageType {
    /** No data. */
    None,
    /** A binary frame. */
    Binary,
    /** A text frame. */
    Text,
    /** An unknown/unhandled frame type. */
    Unknown
  };

 public:
  /**
   * Destructor.
   */
  virtual ~IWebSocket() = default;

  /**
   * Synchronously connects to the endpoint.
   *
   * @see Disconnect
   * @return
   *   - TTV_EC_SUCCESS: The socket has successfully connected and is ready to exchange data.
   *   - TTV_EC_SOCKET_EALREADY: Already connected.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Connect() = 0;

  /**
   * Synchronously disconnects from the endpoint.
   *
   * @see Connect
   * @return
   *   - TTV_EC_SUCCESS: The socket has successfully disconnected.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Disconnect() = 0;

  /**
   * Sends a message of the given type.
   *
   * @param[in] type The type of data represented by the buffer.
   * @param[in] buffer The data to send.
   * @param[in] length The number of bytes in the buffer.
   * @return
   *   - TTV_EC_SUCCESS: The socket has successfully connected and is ready to exchange data.
   *   - TTV_EC_SUCCESS: The message has been sent.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Send(MessageType type, const uint8_t* buffer, size_t length) = 0;

  /**
   * Attempts to receive a message from the web socket.  The next available message will be stored in buffer provided
   * that it is large enough to store it.  Use Peek() to determine how large of a buffer to provide. If the buffer is
   * not large enough to hold the message then an error is returned and the message is not consumed.
   *
   * @see Peek
   * @param[out] type The buffer to store the received bytes in.
   * @param[in] buffer The buffer to store the received bytes in.
   * @param[in] length The largest number of bytes to receive.
   * @param[out] received The number of bytes actually received.
   * @return
   *   - TTV_EC_SUCCESS: Some bytes have been received.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   *   - TTV_EC_INVALID_BUFFER: The given buffer is not big enough to store the next message.
   */
  virtual TTV_ErrorCode Recv(MessageType& type, uint8_t* buffer, size_t length, size_t& received) = 0;

  /**
   * Determines the size and type of the next available message.  If no messages are available then a message type
   * of None will be returned.
   *
   * @see Recv
   * @param[out] type The buffer to store the received bytes in.
   * @param[out] length The number of bytes actually received.
   * @return
   *   - TTV_EC_SUCCESS: Some bytes have been received.
   *   - TTV_EC_SOCKET_ENOTCONN: The socket is not connected.
   *   - TTV_EC_SOCKET_ECONNABORTED: The connected was dropped.
   *   - TTV_EC_SOCKET_ETIMEDOUT: The connection timed out.
   *   - TTV_EC_SOCKET_ECONNRESET: The connection was closed by the remote host.
   *   - TTV_EC_SOCKET_ERR: A generic error occurred.
   */
  virtual TTV_ErrorCode Peek(MessageType& type, size_t& length) = 0;

  /**
   * Determines if the socket is currently connected.
   *
   * @return Whether or not currently connected.
   */
  virtual bool Connected() = 0;
};

/**
 * An interface for classes that can create ISocket instances.
 *
 * @see RegisterSocketFactory
 */
class ttv::ISocketFactory {
 public:
  /**
   * Destructor.
   */
  virtual ~ISocketFactory() = default;

  /**
   * Determines if this factory supports the given protocol.  If the factory claims that is supports the given
   * protocol then it must return a valid ISocket instance from CreateSocket().
   *
   * @param[in] protocol The protocol to check support for.
   * @return true if the protocol is supported, false otherwise.
   */
  virtual bool IsProtocolSupported(const std::string& protocol) = 0;

  /**
   * Attempts to create an ISocket instance that supports the given URI.
   *
   * @see ttv::RegisterSocketFactory
   * @param[in] uri The full uri to create the socket for.
   * @param[out] result The resulting ISocket instance if it was able to be created.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with an ISocket instance to use.
   *   - TTV_EC_UNIMPLEMENTED: This factory does not support the protocol.
   */
  virtual TTV_ErrorCode CreateSocket(const std::string& uri, std::shared_ptr<ISocket>& result) = 0;
};

/**
 * An interface for classes that can create IWebSocket instances.  An IWebSocketFactory generally supports
 * the `ws` and `wss` protocols.
 *
 * @see RegisterWebSocketFactory
 */
class ttv::IWebSocketFactory {
 public:
  /**
   * Destructor.
   */
  virtual ~IWebSocketFactory() = default;

  /**
   * Determines if this factory supports the given protocol.  If the factory claims that is supports the given
   * protocol then it must return a valid ISocket instance from CreateWebSocket().
   *
   * @param[in] protocol The protocol to check support for.
   * @return true if the protocol is supported, false otherwise.
   */
  virtual bool IsProtocolSupported(const std::string& protocol) = 0;

  /**
   * Attempts to create an IWebSocket instance that supports the given URI.
   *
   * @see ttv::RegisterWebSocketFactory
   * @param[in] uri The full uri to create the socket for.
   * @param[out] result The resulting IWebSocket instance if it was able to be created.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with an IWebSocket instance to use.
   *   - TTV_EC_UNIMPLEMENTED: This factory does not support the protocol.
   */
  virtual TTV_ErrorCode CreateWebSocket(const std::string& uri, std::shared_ptr<IWebSocket>& result) = 0;
};

/**
 * A helper class which wraps an ISocket and handles buffering of sent data.  This is useful when sending lots
 * of tiny packets since calling send has a lot of overhead.
 */
class ttv::BufferedSocket {
 public:
  BufferedSocket();
  ~BufferedSocket();

  void Bind(const std::shared_ptr<ISocket>& socket);

  TTV_ErrorCode Connect();
  TTV_ErrorCode Disconnect();

  /**
   * Sends the data over the socket, caching locally if the cache flag is set.  If the cache is full or the
   * the data has been cached for long enough a flush will happen automatically.
   * maxWaitForSend will be used in the case where TTV_EC_SOCKET_EWOULDBLOCK comes back from the socket and we want to
   * try sending again automatically.
   */
  TTV_ErrorCode Send(const uint8_t* buffer, size_t length, bool cache);
  TTV_ErrorCode Recv(uint8_t* buffer, size_t length, size_t& received, uint64_t maxWaitForBufferFill);
  TTV_ErrorCode FlushCache();
  TTV_ErrorCode SetBlockingMode(bool blockingMode);

  uint64_t TotalSent();
  uint64_t TotalReceived();

  /**
   * Computes the average sending bit rate in bits per second over a sliding window.
   */
  TTV_ErrorCode GetAverageSendBitRate(uint64_t measurementWindow, uint64_t& bitsPerSecond) const;

  /**
   * Gives the congestion level, indicating what fraction of time is being spent waiting on sends
   */
  TTV_ErrorCode GetCongestionLevel(uint64_t measurementWindow, double& congestionLevel) const;

  bool Connected();

 private:
  // Buffer up to 65k, which is the maximum size of a TCP packet.
  static const size_t kMaxBufferSize = 0x10000;

  TTV_ErrorCode DoSend(const uint8_t* buffer, size_t length);

  std::shared_ptr<ISocket> mSocket;
  uint64_t mLastFlushTime;
  size_t mCachePos;
  std::array<uint8_t, kMaxBufferSize> mCache;

  SocketTracker mTracker;

  bool mBlocking;
};
