/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/errortypes.h"

#include <deque>
#include <memory>

namespace ttv {
class SocketTracker;
class IMutex;
}  // namespace ttv

/*
 * Class that tracks statistical information about usage of a socket.
 */
class ttv::SocketTracker {
 public:
  SocketTracker();
  virtual ~SocketTracker();

  /*
   * Adds information about usage of a socket
   * @param[in] byteCount The number of bytes sent over the socket
   * @param[in] sentTimestamp The system clock time when the call to send over the socket started.
   * @param[in] blockTime The duration of the blocking send call, in system clock ticks.
   */
  void AddSendInfo(uint32_t byteCount, uint64_t sentTimestamp, uint64_t blockTime);

  /*
   * Clears all the data collected by the socket tracker so far.
   */
  void Reset();

  /*
   * Gets the average outgoing rate of the socket over a sliding window.
   */
  TTV_ErrorCode GetAverageOutgoingRate(uint64_t measurementWindow, uint64_t& bitsPerSecond) const;

  /*
   * Gets the estimated fraction of time spent blocking on the socket over a sliding window. Ranges from 0.0-1.0.
   */
  TTV_ErrorCode GetEstimatedCongestionLevel(uint64_t measurementWindow, double& congestionLevel) const;

 private:
  struct SendEntry {
    uint64_t sentTimestamp;
    uint64_t blockTime;
    uint32_t bytesSent;
  };

  std::deque<SendEntry> mSendEntries;
  std::unique_ptr<IMutex> mMutex;
};
