/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"

#include <sstream>
#include <vector>

namespace ttv {
bool IsValidChannelName(const std::string& channelName);
bool IsValidUserName(const std::string& userName);
bool IsValidOAuthToken(const std::string& oauthToken);
int IsWhitespace(int ch);
std::string ToLowerCase(const std::string& str);
void Trim(std::string& str);
void Split(const std::string& str, std::vector<std::string>& result, char sep, bool includeEmpty);
void Split(const std::string& str, const std::string& delim, std::vector<std::string>& result);
bool StartsWith(const std::string& str, const std::string& prefix);
bool EndsWith(const std::string& str, const std::string& suffix);
bool RFC3339TimeToUnixTimestamp(const std::string& str, Timestamp& result);
std::string UnixTimestampToRFC3339String(Timestamp timestamp);

/**
 * A better string copying function which guarantees null termination of the destination.
 * Some platform versions of snprintf don't handle all cases as expected.
 */
void SafeStringCopy(char* dst, const char* src, size_t maxLen);

/**
 * Generates a 32 character random string.
 */
std::string GetGuid();

std::vector<std::string> ParseArguments(std::string args);

/**
 * Overloaded helper function to parse a numerical value from a string.
 * Uses sscanf from <stdio>
 * http://en.cppreference.com/w/c/io/fscanf
 * http://www.cplusplus.com/reference/cstdio/scanf
 * @return true if number was parsed correctly.
 */
bool ParseNum(const std::string& str, int& out);
bool ParseNum(const std::string& str, signed char& out);
bool ParseNum(const std::string& str, short int& out);
bool ParseNum(const std::string& str, long int& out);
bool ParseNum(const std::string& str, long long int& out);
bool ParseNum(const std::string& str, unsigned int& out);
bool ParseNum(const std::string& str, unsigned char& out);
bool ParseNum(const std::string& str, unsigned short int& out);
bool ParseNum(const std::string& str, unsigned long int& out);
bool ParseNum(const std::string& str, unsigned long long int& out);
bool ParseNum(const std::string& str, float& out);
bool ParseNum(const std::string& str, double& out);
bool ParseNum(const std::string& str, long double& out);
bool ParseNum(const std::string& str, char& out);

/**
 * This function will parse a string value to a bool the following
 * formats are valid true, false, 0, 1
 *
 * @return true if a string is able to be parsed
 */
bool ParseBool(const std::string& str, bool& out);

template <typename PointerType>
std::string PointerToString(PointerType* pointer) {
  std::ostringstream stream;
  stream << reinterpret_cast<void*>(pointer);
  return std::string(stream.str());
}
}  // namespace ttv
