/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <memory>

namespace ttv {
class ISystemClock;

/**
 * Registers the global system clock implementation for the entire SDK.
 */
void SetSystemClock(const std::shared_ptr<ISystemClock>& clock);
/**
 * Retrieves the registered global system clock implementation for the entire SDK.
 */
std::shared_ptr<ISystemClock> GetSystemClock();

/**
 * Retrieves the reported value of ISystemClock::GetFrequency() from the registered system clock.
 */
uint64_t GetSystemClockFrequency();
/**
 * Retrieves the reported value of ISystemClock::GetTime() from the registered system clock.
 * The reported time may not have a consitent base value so it is only useful for relative measurements.
 */
uint64_t GetSystemClockTime();
/**
 * Retrieves the current system time in milliseconds.
 * The reported time may not have a consitent base value so it is only useful for relative measurements.
 */
uint64_t GetSystemTimeMilliseconds();
/**
 * Converts system time to milliseconds.
 */
uint64_t SystemTimeToMs(uint64_t sysTime);
/**
 * Converts milliseconds to system time.
 */
uint64_t MsToSystemTime(uint64_t msTime);

/**
 * Gets the absolute current time as seconds since epoch.
 */
Timestamp GetCurrentTimeAsUnixTimestamp();
}  // namespace ttv

/**
 * The interface for providing time information to SDK components.
 */
class ttv::ISystemClock {
 public:
  virtual ~ISystemClock();

  /**
   * The number of ticks per second.
   */
  virtual uint64_t GetFrequency() = 0;
  /**
   * The current number of system ticks.
   */
  virtual uint64_t GetTime() = 0;
};
