/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/result.h"
#include "twitchsdk/core/task/httptask.h"

namespace {
constexpr const char* kHostName = "https://gql.twitch.tv/gql";

template <class...>
using void_t = void;

template <typename T, typename = void>
struct hasOAuthToken : std::false_type {};

template <typename T>
struct hasOAuthToken<T, void_t<decltype(std::declval<T>().authToken)>> : std::true_type {};
}  // namespace

namespace ttv {
template <typename QueryInfoType>
class GraphQLTask;
}

template <typename QueryInfoType>
class ttv::GraphQLTask : public ttv::HttpTask {
 public:
  using Callback = std::function<void(GraphQLTask* source, Result<typename QueryInfoType::PayloadType>&& result)>;

 public:
  template <typename Type = QueryInfoType>
  GraphQLTask(typename QueryInfoType::InputParams&& input, Callback&& callback,
    typename std::enable_if_t<hasOAuthToken<typename Type::InputParams>::value>* = nullptr)
      : HttpTask(nullptr, nullptr, input.authToken.c_str()),
        mCallback(std::move(callback)),
        mInput(std::move(input)),
        mResult(MakeErrorResult(TTV_EC_API_REQUEST_FAILED)) {
    ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GraphQLTask created");
  }

  template <typename Type = QueryInfoType>
  GraphQLTask(typename QueryInfoType::InputParams&& input, Callback&& callback,
    typename std::enable_if_t<!hasOAuthToken<typename Type::InputParams>::value>* = nullptr)
      : HttpTask(nullptr, nullptr, nullptr),
        mCallback(std::move(callback)),
        mInput(std::move(input)),
        mResult(MakeErrorResult(TTV_EC_API_REQUEST_FAILED)) {
    ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GraphQLTask created");
  }

  virtual void OnComplete() override;

 protected:
  virtual const char* GetTaskName() const override { return QueryInfoType::kTaskName; }

  virtual void FillHttpRequestInfo(HttpRequestInfo& requestInfo) override;
  virtual void ProcessResponse(uint statusCode, const std::vector<char>& response) override;

 private:
  Callback mCallback;
  typename QueryInfoType::InputParams mInput;
  Result<typename QueryInfoType::PayloadType> mResult;
};

template <typename QueryInfoType>
void ttv::GraphQLTask<QueryInfoType>::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = QueryInfoType::kQuery;
  json::ToJson(mInput, root["variables"]);

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

template <typename QueryInfoType>
void ttv::GraphQLTask<QueryInfoType>::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GraphQLTask::ProcessResponse - JSON parsing failed");
      mResult = MakeErrorResult(TTV_EC_WEBAPI_RESULT_INVALID_JSON);
      return;
    }

    const auto& jErrors = jsonVal["errors"];
    if (!jErrors.isNull()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Inside GraphQLTask::ProcessResponse - Error received");
      mResult = MakeErrorResult(TTV_EC_INVALID_GRAPHQL);
      return;
    }

    const auto& jData = jsonVal["data"];
    if (jData.isNull() || !jData.isObject()) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GraphQLTask::ProcessResponse - Invalid GraphQL response");
      mResult = MakeErrorResult(TTV_EC_INVALID_GRAPHQL);
      return;
    }

    typename QueryInfoType::PayloadType payload;
    if (json::ToObject(jData, payload)) {
      mResult = MakeSuccessResult(std::move(payload));
    } else {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GraphQLTask::ProcessResponse - Invalid GraphQL payload");
      mResult = MakeErrorResult(TTV_EC_INVALID_GRAPHQL);
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mResult = MakeErrorResult(TTV_EC_WEBAPI_RESULT_INVALID_JSON);
  }
}

template <typename QueryInfoType>
void ttv::GraphQLTask<QueryInfoType>::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    if (mResult.GetErrorCode() == TTV_EC_API_REQUEST_FAILED && TTV_FAILED(mTaskStatus.ec)) {
      mResult = MakeErrorResult(mTaskStatus.ec);
    }

    mCallback(this, std::move(mResult));
  }
}
