/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <functional>
#include <memory>

namespace ttv {
class IThread;
class IThreadFactory;
class IThreadApi;

using ThreadProc = std::function<void()>;

TTV_ErrorCode SetThreadFactory(std::shared_ptr<IThreadFactory> factory);
TTV_ErrorCode SetThreadApi(std::shared_ptr<IThreadApi> api);

/**
 * Calls Sleep() on the registered ttv::IThreadApi.
 */
void Sleep(uint64_t milliseconds);

TTV_ErrorCode CreateThread(ThreadProc threadProc, const std::string& name, std::shared_ptr<IThread>& result);
}  // namespace ttv

class ttv::IThread {
 public:
  virtual ~IThread();

  /**
   * Begins the thread.  This can only be called once.
   */
  virtual void Run() = 0;
  virtual bool Joinable() const = 0;
  virtual void Join() = 0;
};

class ttv::IThreadFactory {
 public:
  virtual ~IThreadFactory();

  virtual TTV_ErrorCode CreateThread(
    ThreadProc threadProc, const std::string& name, std::shared_ptr<IThread>& result) = 0;
};

class ttv::IThreadApi {
 public:
  virtual ~IThreadApi();

  /**
   * Sleeps the current thread for at least the given number of milliseconds.
   */
  virtual void Sleep(uint64_t milliseconds) = 0;
};
