/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/eventqueue.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <atomic>

namespace ttv {
class ThreadedEventScheduler;
class EventQueue;
}  // namespace ttv

/**
 * An implementation of IEventScheduler that runs and drives itself in a separate background thread.
 *
 * Shutdown() should be called before the ThreadedEventScheduler is destroyed.
 * Additionally, clients should wait for the shutdownTask lambda to be called back before destroying the
 * ThreadedEventScheduler.
 *
 * ScheduleTask() and CancelTask() should not be called after ThreadedEventScheduler is shut down. Otherwise, it is
 * expected that TTV_EC_NOT_INITIALIZED is returned.
 */
class ttv::ThreadedEventScheduler : public IEventScheduler {
 public:
  ThreadedEventScheduler();
  ThreadedEventScheduler(const std::string& name);
  virtual ~ThreadedEventScheduler() override;

  // IEventScheduler overrides
  virtual Result<TaskId> ScheduleTask(TaskParams&& taskParams) override;
  virtual TTV_ErrorCode CancelTask(TaskId taskId) override;
  virtual TTV_ErrorCode Shutdown(TaskFunc&& shutdownTask) override;
  virtual EventSchedulerState GetState() override;

 private:
  void StartThreadProc();
  void EventSchedulerThreadProc();

 private:
  std::shared_ptr<IThread> mEventSchedulerThread;
  EventQueue mEventQueue;
  TaskFunc mShutDownTask;
  std::string mName;
  std::atomic<EventSchedulerState> mState;
};
