/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/types/coretypes.h"

#include <atomic>
#include <memory>
#include <vector>

namespace ttv {
class WaitForEventWithTimeout;
class WaitForExpiry;
class RetryBackoffTable;
class RetryTimer;
class LambdaRetryTimer;

uint64_t JitterTime(uint64_t baseMs, uint64_t width);
}  // namespace ttv

class ttv::WaitForEventWithTimeout {
 public:
  enum eWaitState {
    WFEWT_Waiting,
    WFEWT_Complete,
    WFEWT_TimedOut,
  };

 public:
  WaitForEventWithTimeout();

  void Reset(uint64_t timeout);
  void Complete();
  eWaitState GetState();

  bool IsTimedOut() { return GetState() == WFEWT_TimedOut; }
  bool IsComplete() { return GetState() == WFEWT_Complete; }

 private:
  uint64_t mExpiryTime;
  eWaitState mState;
};

class ttv::WaitForExpiry {
 public:
  WaitForExpiry();

  void Set(uint64_t milliseconds);
  void SetWithJitter(uint64_t milliseconds, uint64_t jitterWidthMs);
  void AdjustDuration(uint64_t milliseconds);
  bool Check(bool clearWhenExpired);
  void Clear();
  bool IsSet() const { return mStartTime != 0; }

  uint64_t GetStartTime() const { return mStartTime; }
  uint64_t GetEndTime() const { return mEndTime; }
  uint64_t GetRemainingTime() const;
  uint64_t GetElapsedTime() const;

 private:
  uint64_t mStartTime;
  uint64_t mEndTime;
};

/**
 * Maintains a table of intervals used to manage the backoff that should be done
 * when retrying requests to a failing endpoint.
 * Usage:
 *   Create the table with the desired backoff.
 *   When the timed operation succeeds, call Reset() to reset the internal backoff.
 *   When the timed operation fails, call GetInterval() to start your timer and call Advance() so the
 *     next call to GetInterval() will use the next interval.
 */
class ttv::RetryBackoffTable {
 public:
  /**
   * Creates a default table with default jitter.
   */
  RetryBackoffTable();
  /**
   * Uses an explicit table and jitter.
   */
  RetryBackoffTable(const std::vector<uint64_t>& tableMilliseconds, uint64_t retryJitterWidthMs);
  /**
   * Computes an exponential backoff table and uses the given jitter.
   */
  RetryBackoffTable(uint64_t maxInterval, uint64_t retryJitterWidthMs);

  /**
   * Advance to the next retry interval.
   */
  void Advance();
  /**
   * Reset the retry attempt index.
   */
  void Reset();
  /**
   * Gets the next timer interval to use.
   */
  uint64_t GetInterval() const;

 private:
  void CreateTable(uint64_t maxInterval);

  std::vector<uint64_t> mBackOffTableMilliseconds;
  uint64_t mJitterMilliseconds;
  uint32_t mNextAttemptNumber;
};

/**
 * Manages the timing of requests or connections to backend services.  Supports backoff as attempts continue to fail.
 *
 * Call ScheduleNextRetry() to start the timer.
 * Call CheckNextRetry() periodically to see if it's time to retry.  Once this returns true it will not return true
 * again until ScheduleNextRetry() is called. Call Clear() to stop the retry timer and reset the backoff table if not
 * managing a persisten connection.
 *
 * Optionally, if the timer is managing a persistent connection then the backoff table reset can be controlled using the
 * following methods. Call StartGlobalReset() when successfully connected to begin the timer that will reset the
 * backoff. Call CheckGlobalReset() periodically to apply the backoff reset when it's time. Call ClearGlobalReset() when
 * CheckGlobalReset() has elapsed or the request completes sucessfully / the connection is closed purposefully.
 */
class ttv::RetryTimer {
 public:
  /**
   * Configures the timer with a default table.
   */
  RetryTimer();
  /**
   * Configures the timer with an explicit table.
   */
  RetryTimer(const std::vector<uint64_t>& backOffTableMs, uint64_t retryJitterWidthMs);
  /**
   * Configures the timer with exponential backoff up to the given interval.
   */
  RetryTimer(uint64_t maxInterval, uint64_t retryJitterWidthMs);

  void SetBackoffTable(const std::vector<uint64_t>& backOffTableMs, uint64_t retryJitterWidthMs);
  void SetBackoffTable(uint64_t maxInterval, uint64_t retryJitterWidthMs);

  /**
   * Marks the current retry as complete and advances the retry timer to the next attempt.
   */
  void ScheduleNextRetry();
  /**
   * Used to determine if it's time to retry.  When the result of the attempt is complete be sure to call
   * RetryComplete().
   */
  bool CheckNextRetry();
  /**
   * Stops the retry timer and resets the backoff table.  This should be called when done with the timer and not
   * managing a persistent connection.
   */
  void Clear();

  void StartGlobalReset(uint64_t milliseconds);
  bool CheckGlobalReset();
  void ClearGlobalReset();

  bool IsRetrySet() const { return mNextRetry.IsSet(); }
  bool IsGlobalSet() const { return mGlobalResetTimer.IsSet(); }

 protected:
  uint64_t GetNextAttempt();

  WaitForExpiry mNextRetry;
  WaitForExpiry mGlobalResetTimer;  //!< The timer which will reset mNextAttemptNumber.
  std::vector<uint64_t> mBackOffTable;
  uint64_t mRetryJitter;
  uint32_t mNextAttemptNumber;
};

/**
 * Implementation of a retry timer that uses an IEventScheduler to schedule lambdas/tasks.
 *
 * SetCallback() and SetEventScheduler() should be called first to set up the timer.
 * Start() will call the callback after a delay on the IEventScheduler.
 *
 * The lifetime of LambdaRetryTimer determines the lifetime of its associated callback.
 * Destroying the LambdaRetryTimer will cancel its associated callback on the IEventScheduler.
 */
class ttv::LambdaRetryTimer {
 public:
  using CallbackFunc = std::function<void()>;

 public:
  /**
   * Configures the timer with a default table.
   */
  LambdaRetryTimer();
  /**
   * Configures the timer with an explicit table.
   */
  LambdaRetryTimer(const std::vector<uint64_t>& backOffTableMs, uint64_t retryJitterWidthMs);
  /**
   * Configures the timer with exponential backoff up to the given interval.
   */
  LambdaRetryTimer(uint64_t maxInterval, uint64_t retryJitterWidthMs);
  /**
   * Destroying the LambdaRetryTimer will cancel any callbacks already scheduled on the associated IEventScheduler
   */
  ~LambdaRetryTimer();
  /**
   * Set the callback to run when the timer ends.
   */
  void SetCallback(CallbackFunc&& func);
  /**
   * Set the IEventScheduler to schedule callbacks on.
   */
  void SetEventScheduler(const std::shared_ptr<IEventScheduler>& eventScheduler);
  /**
   * Starts as a normal timer with the given timeout - run the callback after `milliseconds`.
   * @return
   *   - TTV_EC_SUCCESS: The timer was successfully started.
   *   - TTV_EC_NOT_INITIALIZED: The LambdaRetryTimer or its IEventScheduler has not yet been initialized.
   */
  TTV_ErrorCode Start(uint64_t milliseconds);
  /**
   * Starts the timer in backoff mode, using the internal table to get the interval.
   * @return
   *   - TTV_EC_SUCCESS: The timer was successfully started.
   *   - TTV_EC_NOT_INITIALIZED: The LambdaRetryTimer or its IEventScheduler has not yet been initialized.
   */
  TTV_ErrorCode StartBackoff();
  /**
   * Stops the timer and cancels the callback on the IEventScheduler
   * @return
   *   - TTV_EC_SUCCESS: The task was successly stopped.
   *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has not yet been initialized or been shutdown.
   *   - TTV_EC_OPERATION_FAILED: The timer was not running and is already current stopped.
   */
  TTV_ErrorCode Stop();
  /**
   * Resets the internal backoff counter.
   */
  void ResetBackoff();
  /**
   * Returns whether or not the timer is set or not.
   */
  bool IsSet() const;

 protected:
  std::shared_ptr<IEventScheduler> mEventScheduler;
  RetryBackoffTable mBackOffTable;
  CallbackFunc mCallback;
  TaskId mTaskId;
  std::atomic_bool mTimerSet;
};
