/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/types/tracingtypes.h"

#include <stdarg.h>
#include <unordered_map>

#include <memory>
#include <string>
#include <vector>

namespace ttv {
class IMutex;
class ITracer;
class TracerBase;
class AutoTracer;

void SetTracer(std::shared_ptr<ITracer> tracer);

namespace trace {
void Message(const char* component, const MessageLevel messageLevel, const char* format, ...);
void MessageVaList(const char* component, const MessageLevel messageLevel, const char* format, va_list args);

TTV_ErrorCode SetOutputFile(const std::string& path);

TTV_ErrorCode SetComponentMessageLevel(const char* component, MessageLevel level);
TTV_ErrorCode GetComponentMessageLevel(const char* component, MessageLevel& level);

TTV_ErrorCode SetGlobalMessageLevel(MessageLevel level);
TTV_ErrorCode GetGlobalMessageLevel(MessageLevel& level);
}  // namespace trace
}  // namespace ttv

class ttv::ITracer {
 public:
  virtual ~ITracer() = default;

  virtual void Message(const char* component, const MessageLevel messageLevel, const char* format, ...) = 0;
  virtual void MessageVaList(
    const char* component, const MessageLevel messageLevel, const char* format, va_list args) = 0;

  virtual TTV_ErrorCode SetOutputFile(const std::string& path) = 0;

  virtual TTV_ErrorCode SetComponentMessageLevel(const char* component, MessageLevel level) = 0;
  virtual TTV_ErrorCode GetComponentMessageLevel(const char* component, MessageLevel& level) = 0;

  virtual TTV_ErrorCode SetGlobalMessageLevel(MessageLevel level) = 0;
  virtual TTV_ErrorCode GetGlobalMessageLevel(MessageLevel& level) = 0;
};

class ttv::TracerBase : public ITracer {
 public:
  TracerBase();
  virtual ~TracerBase() override;

  virtual void Message(const char* component, const MessageLevel messageLevel, const char* format, ...) override;
  virtual void MessageVaList(
    const char* component, const MessageLevel messageLevel, const char* format, va_list args) override;

  virtual TTV_ErrorCode SetOutputFile(const std::string& path) override;

  virtual TTV_ErrorCode SetComponentMessageLevel(const char* component, MessageLevel level) override;
  virtual TTV_ErrorCode GetComponentMessageLevel(const char* component, MessageLevel& level) override;

  virtual TTV_ErrorCode SetGlobalMessageLevel(MessageLevel level) override;
  virtual TTV_ErrorCode GetGlobalMessageLevel(MessageLevel& level) override;

 protected:
  bool ShouldLog(const char* component, const MessageLevel messageLevel);
  void GetLinePrefix(const char* component, const char* messageLevel, char* buffer, size_t bufferLength);

  // Performs the actual logging to the output.
  virtual void Log(const char* component, const char* messageLevel, const char* format, va_list args) = 0;

  virtual bool OpenFile(const std::string& path);
  virtual void LogToFile(const char* message);
  virtual bool CloseFile();

  std::vector<char> mPrintBuffer;
  std::unordered_map<std::string, MessageLevel> mTraceLevels;
  MessageLevel mGlobalTraceLevel;
  std::unique_ptr<IMutex> mMutex;  //!< Tries to keep logging on different threads done atomically.
  FILE* mTraceFile;
};

class ttv::AutoTracer {
 public:
  AutoTracer(const char* component, const MessageLevel messageLevel, const char* message)
      : mComponent(component), mMessageLevel(messageLevel), mMessage(message) {
    ttv::trace::Message(mComponent, mMessageLevel, "Entering %s",
      mMessage);  //-V111   Call of function ____ with variable number of arguments. Fourth argument has memsize type.
  }
  ~AutoTracer() {
    ttv::trace::Message(mComponent, mMessageLevel, "Exiting %s",
      mMessage);  //-V111   Call of function ____ with variable number of arguments. Fourth argument has memsize type.
  }

 private:
  const char* mComponent;
  MessageLevel mMessageLevel;
  const char* mMessage;
};
