/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"

namespace ttv {
class IMutex;
class TrackingContext;
}  // namespace ttv

/**
 * A class to send tracking events with shared properties.
 *
 * Shared properties can be set on this class, which will be added to any tracking events sent through
 * this class. Tracking contexts can be a root context, which direcly sends the tracking events, or a
 * child context, which forwards the events to its parent. As such, a child context inherits the shared
 * properties of its parent.
 *
 * Shared properties in a child context override the values of properties in the parent context with the
 * same key. Similarly, properties directly passed into TrackEvent override any shared properties with the
 * same key.
 */
class ttv::TrackingContext {
 public:
  /**
   * Create a root tracking context that directly sends tracking events.
   */
  TrackingContext();

  /**
   * Create a child tracking context, which sends events to its parent.
   * @param[in] parent The parent tracking context.
   */
  TrackingContext(const std::shared_ptr<TrackingContext>& parent);

  /**
   * Sets a property on the context.
   */
  void SetProperty(const std::string& key, const TrackingValue& value);

  /**
   * Sets multiple properties on the context. These are guaranteed to be atomically set.
   */
  void SetProperties(std::initializer_list<std::pair<std::string, TrackingValue>> properties);

  /**
   * Track an event.
   * @param[in] eventName The name of the event to be tracked.
   * @param[in] properties A map of properties to send with the event.
   */
  TTV_ErrorCode TrackEvent(const std::string& eventName, std::map<std::string, TrackingValue>&& properties);

 private:
  std::unique_ptr<ttv::IMutex> mMutex;
  std::shared_ptr<TrackingContext> mParentContext;
  std::map<std::string, TrackingValue> mProperties;
};
