/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <stdint.h>

#include <functional>
#include <string>
#include <vector>

using uint = unsigned int;

namespace ttv {
using UserId = uint32_t;
using ChannelId = uint32_t;
using GameId = uint32_t;

// Represents an absolute time in seconds since epoch.
using Timestamp = uint32_t;
using Color = uint32_t;
using BroadcastId = uint32_t;

using TaskFunc = std::function<void()>;
using TaskId = uint64_t;

struct UserInfo {
  UserInfo();

  bool operator==(const UserInfo& other) const;
  bool operator!=(const UserInfo& other) const;

  std::string userName;
  std::string displayName;
  std::string bio;
  std::string logoImageUrl;
  UserId userId;
  Timestamp createdTimestamp;
};

/**
 * Information about a Twitch channel.  This is a snapshot and values may change over time.
 */
struct ChannelInfo {
  ChannelInfo();

  using IdType = ChannelId;
  IdType GetId() const { return channelId; }

  std::string displayName;
  std::string name;
  std::string game;
  std::string description;
  std::string status;
  std::string language;
  std::string broadcasterLanguage;
  std::string logoImageUrl;
  std::string channelUrl;
  std::string videoBannerImageUrl;
  std::string profileBannerImageUrl;

  ChannelId channelId;
  Timestamp createdAtTimestamp;
  Timestamp updatedAtTimestamp;
  Timestamp lastBroadcastEndTimestamp;
  uint32_t numFollowers;
  uint32_t numViews;

  bool mature;
  bool partner;
  bool affiliate;
};

/**
 * PreviewImages - URLs for preview images (of different sizes).
 */
struct PreviewImages {
  std::string largeUrl;
  std::string mediumUrl;
  std::string smallUrl;
  std::string templateUrl;
};

/**
 * BroadcastPlaform - Broadcast platform of a given stream.
 * https://git.xarth.tv/web/jax/blob/master/db/query/filter.go#L715
 */
enum class BroadcastPlatform { WatchParty, Premiere, Rerun, Playlist, Mobile, Xbox, PS4, Live, Unknown };

/**
 * StreamType - Type of a given stream.
 * Is deprecated in favor of BroadcastPlatform on the back-end.
 * https://git.xarth.tv/web/jax/blob/master/db/query/filter.go#L715
 */
enum class StreamType { WatchParty, Premiere, Rerun, Playlist, Live, Unknown };

/**
 * Information about a live stream.
 */
struct StreamInfo {
  StreamInfo();

  ChannelInfo channelInfo;
  PreviewImages previewImages;
  std::string game;

  double averageFPS;
  uint64_t streamId;
  uint64_t archiveVideoId;
  uint64_t delay;
  uint64_t viewerCount;
  uint32_t videoHeight;
  Timestamp createdAtTimestamp;

  BroadcastPlatform broadcastPlatform;
  StreamType streamType;
  bool isPlaylist;
};

/**
 * Holds values for a channel's updated stream information.
 */
struct StreamInfoUpdate {
  StreamInfoUpdate();

  std::string title;
  std::string game;
  GameId gameId;
};

enum class PubSubState { Disconnected, Connecting, Connected, Disconnecting };

/**
 * VodStatus - The current status for a VOD object on the backend.
 */
enum class VodStatus {
  Recording,
  Recorded,  //!< The VOD is still being recorded and the stream is live.
  Unknown    //!< The VOD has finished being recorded and is available.

};

/**
 * VodType - The type of VOD represented.
 */
enum class VodType { Highlight, Archive, Upload, Unknown };

/**
 * WatchPartyUpdate - Info received about a watch party occuring on a channel.
 */
struct WatchPartyUpdate {
  WatchPartyUpdate();

  std::string incrementUrl;  //!< The URL to increment whenever a new VOD is played.
  std::string vodId;         //!< The currently playing VOD ID
  std::string title;         //!< The title of the currently playing VOD.
  std::string watchPartyId;  //!< The unique ID for this watch party (should be the same across all VODs)
  VodType broadcastType;     //!< The type of vod being played
  bool viewable;  //!< Marked true if this VOD is published and available for anyone to view, false if it is not.
};

/**
 * ProfileImage - Info about a new profile image.
 */
struct ProfileImage {
  std::string url;     //!< The URL of the profile image.
  std::string format;  //!< The filetype/format of the profile image.
  uint32_t width;      //!< Width of the profile image.
  uint32_t height;     //!< Height of the profile image.
};

/**
 * SquadStatus - The possible states of a squad stream
 */
enum class SquadStatus { Unknown, Pending, Live, Ended };

/**
 * SquadMember - The information on a member of a squad stream
 */
struct SquadMember {
  SquadMember();

  std::string userLogin;
  std::string userDisplayName;
  std::string profileImageUrl150;  //!< The url of the user's profile image, size of 150x150.
  ChannelId channelId;
};

/**
 * SquadInfo - The information on a squad stream, received through pubsub.
 */
struct SquadInfo {
  SquadInfo();

  std::vector<SquadMember> members;
  std::string squadId;
  UserId ownerId;
  SquadStatus status;
};

/**
 * CanTheyError - This holds the information that is returned from the
 * CanThey service during backend requests
 */
struct CanTheyError {
  std::string code;
  std::string message;
  std::vector<std::string> links;
};

/**
 * TrackingValue - A class that can contain several types of values to send as tracking properties.
 */
class TrackingValue {
 public:
  /**
   * The type of value contained.
   */
  enum class Type { Null, Boolean, Integer, Double, String };

  /**
   * Default constructor and nullptr create a value of Null type.
   */
  TrackingValue();
  TrackingValue(std::nullptr_t value);

  /**
   * Creates a value of boolean type.
   */
  TrackingValue(bool value);

  /**
   * Creates a value of integer type.
   */
  TrackingValue(int64_t value);
  TrackingValue(uint32_t value);
  TrackingValue(int value);

  /**
   * Creates a value of double type.
   */
  TrackingValue(double value);

  /**
   * Creates a value of string type.
   */
  TrackingValue(const std::string& value);
  TrackingValue(std::string&& value);
  TrackingValue(const char* value);

  TrackingValue(const TrackingValue& src);
  TrackingValue(TrackingValue&& src);

  ~TrackingValue();

  TrackingValue& operator=(const TrackingValue& src);
  TrackingValue& operator=(TrackingValue&& src);

  /**
   * Returns the type of value contained.
   */
  Type GetType() const { return mType; }

  /**
   * Returns the value contained.
   * It is illegal to call the getter that does not match the contained type. As such, the user of this interface
   * should always call GetType() first to determine what value type is stored.
   */
  bool GetBooleanValue() const;
  int64_t GetIntegerValue() const;
  double GetDoubleValue() const;
  std::string GetStringValue() const;

 private:
  union Value {
    Value() {}
    ~Value() {}

    bool asBool;
    int64_t asInteger;
    double asDouble;
    std::string asString;
  } mValue;

  Type mType;
};
}  // namespace ttv
