/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/value.h"

#include <functional>
#include <string>
#include <vector>

/**
 * The unique module identifier for core.
 */
const uint32_t TTV_MODULE_ID_TWITCH_CORE = 0;

/**
 * The generic error code type which represents error codes from all modules.  It is constructed piecemeal as follows:
 *
 * `ModuleId(16) | ErrorId(16)`
 *
 * - ModuleId: 16 bits to represent the module the code is defined in.  This is not necessarily the source of the error.
 * Each module has a unique id.
 * - ErrorId:  16 bits to represent the unique error code within the module.  Each module generally has an enumeration
 * which defines them without the flag or module id.
 */
using TTV_ErrorCode = uint32_t;

const uint32_t kErrorIdMask = 0xFFFF;

/**
 * Creates an error code integer conforming to the format of TTV_ErrorCode.
 */
#define CONSTRUCT_ERROR_ID_VALUE(MODULE_ID, ERROR_ID) ((MODULE_ID << 16) | (ERROR_ID & kErrorIdMask))

/**
 * Extracts the module id from the given TTV_ErrorCode.
 */
#define GET_MODULE_ID(ec) ((ec >> 16))

/**
 * Extracts the error id from the given TTV_ErrorCode.  It is appropriate to cast the id to the module-specific
 * enumeration value.
 */
#define GET_ERROR_ID(ec) (ec & kErrorIdMask)

namespace ttv {
struct EnumValue;
struct ErrorDetails;

/**
 * Function signature for getting a human readable string value for an error code.
 */
using ErrorToStringFunction = const char* (*)(TTV_ErrorCode ec);
/**
 * Function signature for getting available error codes.
 */
using ErrorCodeValueFunction = void (*)(std::vector<EnumValue>& result);
/**
 * Registers an error code to string function.  Each module should register a function for its error codes.
 */
void RegisterErrorToStringFunction(ErrorToStringFunction func);
/**
 * Unregisters a function previously registered via UnregisterErrorToStringFunction().
 */
void UnregisterErrorToStringFunction(ErrorToStringFunction func);
/**
 * Registers a function which will populate error code values.
 */
void RegisterErrorCodeValueFunction(ErrorCodeValueFunction func);
/**
 * Unregisters a function which will populate error code values.
 */
void UnregisterErrorCodeValueFunction(ErrorCodeValueFunction func);
}  // namespace ttv

#define TTV_CORE_ERROR_IDS                                                                                                                                                                                  \
  /* Success */                                                                                                                                                                                             \
  USE_TTV_SUCCESS_ID(TTV_EC_SUCCESS)                                                                                                                                                                        \
                                                                                                                                                                                                            \
  /* Errors */                                                                                                                                                                                              \
  BEGIN_ERROR_IDS(TTV_EC_BEGIN_ERRORS)                                                                                                                                                                      \
  USE_TTV_ERROR_ID(TTV_EC_UNKNOWN_ERROR)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_OPEN_FILE)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_NOTENOUGHDATA)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_MUTEX_NOT_ACQUIRED)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_CONDITION_WAIT_FAILED)                                                                                                                                                            \
  USE_TTV_ERROR_ID(TTV_EC_WAIT_TIMEOUT)                                                                                                                                                                     \
  USE_TTV_ERROR_ID(TTV_EC_OAUTH_SCOPES_MISSING)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_ALREADY_INITIALIZED)                                                                                                                                                              \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_WRITE_TO_FILE)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_CREATE_MUTEX)                                                                                                                                                              \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_DESTROY_MUTEX)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_COULD_NOT_TAKE_MUTEX)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_COULD_NOT_RELEASE_MUTEX)                                                                                                                                                          \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_MUTEX)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_ARG)                                                                                                                                                                      \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_FORMAT)                                                                                                                                                                   \
  USE_TTV_ERROR_ID(TTV_EC_NOT_INITIALIZED)                                                                                                                                                                  \
  USE_TTV_ERROR_ID(TTV_EC_AUTHENTICATION)                                                                                                                                                                   \
  USE_TTV_ERROR_ID(TTV_EC_FORBIDDEN)                                                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_OAUTHTOKEN)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_USERINFO_NOT_AVAILABLE)                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_NOT_AVAILABLE)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_PARAMETERS_CHANGED)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_MEMORY)                                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_ALIGN16_REQUIRED)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_HTTPREQUEST_ERROR)                                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_RATE_LIMITED)                                                                                                                                                                     \
  USE_TTV_ERROR_ID(TTV_EC_NO_FACTORIES_REGISTERED)                                                                                                                                                          \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_CLIENTID)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_CHANNEL_ID)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_OPERATION_FAILED)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_API_REQUEST_FAILED)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_API_REQUEST_TIMEDOUT)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_HTTP_REQUEST_PARAMS)                                                                                                                                                      \
  USE_TTV_ERROR_ID(TTV_EC_COINITIALIZE_FAIED)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_INVALID_JSON)                                                                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_AUTHTOKEN)                                                                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_STREAMKEY)                                                                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_CHANNELNAME)                                                                                                                                                     \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_INGESTS)                                                                                                                                                         \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_RECORDING_STATUS)                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_STREAMINFO)                                                                                                                                                      \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_INVALID_VIEWERS)                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_USERNAME)                                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_WEBAPI_RESULT_NO_USER_DISPLAY_NAME)                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_NEED_TO_LOGIN)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_LOGIN)  /*!< Invalid login credentials. */                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_BANNED_USER)    /*!< User is banned */                                                                                                                                            \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_USERID) /*!< Invalid user name. */                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_BUFFER)                                                                                                                                                                   \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_CALLBACK)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_JSON)                                                                                                                                                                     \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_STATE)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_SUSPEND_THREADSYNC)                                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_CANNOT_SIGNAL_THREADSYNC)                                                                                                                                                         \
  USE_TTV_ERROR_ID(TTV_EC_REQUEST_TIMEDOUT)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_REQUEST_PENDING)                                                                                                                                                                  \
  /*****************************/                                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_OS_TOO_OLD)                                                                                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_SHUTTING_DOWN)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_SHUT_DOWN)                                                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_UNIMPLEMENTED)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_ALREADY_LOGGED_IN)                                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_ALREADY_CACHED)                                                                                                                                                                   \
  USE_TTV_ERROR_ID(TTV_EC_FEATURE_DISABLED)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_UNSUPPORTED)                                                                                                                                                                      \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_INSTANCE)                                                                                                                                                                 \
  /* Socket errors */                                                                                                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_EWOULDBLOCK)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_ENOTCONN)                                                                                                                                                                  \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_TRY_AGAIN)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_CONNECT_FAILED)                                                                                                                                                            \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_ECONNABORTED)                                                                                                                                                              \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_EALREADY)                                                                                                                                                                  \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_CREATE_FAILED)                                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_GETADDRINFO_FAILED)                                                                                                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_HOST_NOT_FOUND)                                                                                                                                                            \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_RECV_ERROR)                                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_SEND_ERROR)                                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_IOCTL_ERROR)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_ETIMEDOUT)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_ECONNRESET)                                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_SOCKET_ERR)                                                                                                                                                                       \
  /*****************************/                                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_PUBSUB_BAD_TOPIC)                                                                                                                                                                 \
  USE_TTV_ERROR_ID(TTV_EC_PUBSUB_BAD_MESSAGE)                                                                                                                                                               \
  USE_TTV_ERROR_ID(TTV_EC_PUBSUB_SERVER_INTERNAL_ERROR)                                                                                                                                                     \
  USE_TTV_ERROR_ID(TTV_EC_PUBSUB_RESPONSE_ERROR)                                                                                                                                                            \
  /*****************************/                                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_PROFILEIMAGE_IMAGE_VALIDATION_FAILED)                                                                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_PROFILEIMAGE_FORMAT_VALIDATION_FAILED)                                                                                                                                            \
  USE_TTV_ERROR_ID(TTV_EC_PROFILEIMAGE_SIZE_VALIDATION_FAILED)                                                                                                                                              \
  USE_TTV_ERROR_ID(TTV_EC_PROFILEIMAGE_BACKEND_FAILURE)                                                                                                                                                     \
  /*****************************/                                                                                                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_GRAPHQL_ERROR)                                                                                                                                                                    \
  USE_TTV_ERROR_ID(TTV_EC_INVALID_GRAPHQL)                                                                                                                                                                  \
  /**/                                                                                                                                                                                                      \
  USE_TTV_ERROR_ID(                                                                                                                                                                                         \
    TTV_EC_INVALID_STRUCT_SIZE) /*!< The size field of the struct does not have the expected value.  Some fields may have been added to the struct and you may have to set them and recompile your code. */ \
  USE_TTV_ERROR_ID(TTV_EC_REQUEST_ABORTED) /*!< The request was aborted and did not finish. */                                                                                                              \
  USE_TTV_ERROR_ID(TTV_EC_STILL_IN_USE)    /*!< The requested feature is still in use. */                                                                                                                   \
  END_ERROR_IDS(TTV_EC_END_ERRORS)

#define USE_TTV_ERROR_ID(ec) ec,
#define USE_TTV_SUCCESS_ID(ec) ec = 0,
#define BEGIN_ERROR_IDS(ec) ec = CONSTRUCT_ERROR_ID_VALUE(TTV_MODULE_ID_TWITCH_CORE, 1),
#define END_ERROR_IDS(ec) ec

/**
 * TTV_CoreErrorId - The error codes defined in core.
 */
enum TTV_CoreErrorId { TTV_CORE_ERROR_IDS };

#undef BEGIN_ERROR_IDS
#undef END_ERROR_IDS
#undef USE_TTV_SUCCESS_ID
#undef USE_TTV_ERROR_ID

/**
 * Determines if the given code does not have the error bit set.  Pure success and warnings will pass this check.
 */
#define TTV_SUCCEEDED(ec) ((ec) == TTV_EC_SUCCESS)
/**
 * Determines if the given code has the error bit set.
 */
#define TTV_FAILED(ec) ((ec) != TTV_EC_SUCCESS)
/**
 * Asserts that the given error code passes TTV_SUCCEEDED(ec).
 */
#define ASSERT_ON_ERROR(ec)        \
  do {                             \
    TTV_ASSERT(TTV_SUCCEEDED(ec)); \
  } while (0)
#define RETURN_CLIENT_ERROR(ec) \
  do {                          \
    return ec;                  \
  } while (0)
#define TTV_RETURN_ON_NULL(ptr, ec) \
  do {                              \
    if ((ptr) == nullptr)           \
      RETURN_CLIENT_ERROR(ec);      \
  } while (0)
#define TTV_RETURN_ON_NOT_NULL(ptr, ec) \
  do {                                  \
    if ((ptr) != nullptr)               \
      RETURN_CLIENT_ERROR(ec);          \
  } while (0)
#define TTV_RETURN_ON_EMPTY_STRING(str, ec) \
  do {                                      \
    if (str == nullptr || str[0] == '\0')   \
      RETURN_CLIENT_ERROR(ec);              \
  } while (0)
#define TTV_RETURN_ON_EMPTY_STDSTRING(str, ec) \
  do {                                         \
    if (str.size() == 0)                       \
      RETURN_CLIENT_ERROR(ec);                 \
  } while (0)
#define TTV_RETURN_ON_ERROR(err) \
  do {                           \
    if (TTV_FAILED(ec))          \
      RETURN_CLIENT_ERROR(ec);   \
  } while (0)
#define TTV_RETURN_ON_DIFFERENT(a, b, ec) \
  do {                                    \
    if ((a) != (b))                       \
      RETURN_CLIENT_ERROR(ec);            \
  } while (0)
#define TTV_RETURN_ON_SAME(a, b, ec) \
  do {                               \
    if ((a) == (b))                  \
      RETURN_CLIENT_ERROR(ec);       \
  } while (0)

namespace ttv {
/**
 * The central error code string lookup function to be used by clients which want a human readable string from a generic
 * error code.
 */
const char* ErrorToString(TTV_ErrorCode ec);

/**
 * The error code string lookup for core.  This is not normally used directly by clients.  Use ttv::ErrorToString()
 * instead.
 */
const char* CoreErrorToString(TTV_ErrorCode ec);

void GetCoreErrorCodeValues(std::vector<EnumValue>& result);

void GetAllErrorCodes(std::vector<EnumValue>& result);
}  // namespace ttv

struct ttv::EnumValue {
  std::string name;
  uint32_t value;

  EnumValue(const char* name, uint32_t value);
};

/**
 * A structure that provides extra information about an Error in the SDK. This may still be used to return
 * TTV_EC_SUCCESS.
 */
struct ttv::ErrorDetails {
  ErrorDetails();
  explicit ErrorDetails(TTV_ErrorCode errorCode);
  explicit ErrorDetails(TTV_ErrorCode errorCode, const std::string& message);

  /**
   * The collection of details about the error.
   */
  ttv::json::Value details;
  /**
   * The error code associated with the error.
   */
  TTV_ErrorCode ec;

  json::Value& operator[](const char* const key);
  ErrorDetails& operator=(TTV_ErrorCode other);
};
