/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/oauthtoken.h"

#include <map>
#include <memory>

namespace ttv {
class TaskRunner;
class User;
struct UserInfo;
class IUserListener;
class UserRepository;
class GetUserTask;
}  // namespace ttv

class ttv::User : public Component, public std::enable_shared_from_this<ttv::User> {
 public:
  User(UserId userId);
  virtual ~User() override;

  void SetUserRepository(const std::shared_ptr<UserRepository>& repository) { mUserRepository = repository; }

  void AddListener(const std::shared_ptr<IUserListener>& listener);
  void RemoveListener(const std::shared_ptr<IUserListener>& listener);

  std::shared_ptr<ComponentContainer> GetComponentContainer() { return mComponentContainer; }

  /**
   * Assigns a new OAuth token to the user and initiates validation.  When validation is complete it will fire
   * IUserListener::OnUserLogInComplete.
   */
  void LogIn();
  /**
   * Resets the OAuth token and fires IUserListener::OnUserLogOutComplete.
   */
  void LogOut();
  /**
   * Called by any client of User the has attempted to use the OAuth token to access a resource and was rejected.
   */
  void ReportOAuthTokenInvalid(const std::shared_ptr<const OAuthToken>& oauthToken, TTV_ErrorCode ec);

  const UserInfo& GetUserInfo() const { return mUserInfo; }
  void SetUserInfo(const UserInfo& userInfo);

  std::string GetUserName() const { return mUserInfo.userName; }

  void SetOAuthToken(const std::shared_ptr<OAuthToken>& authToken);
  std::shared_ptr<const OAuthToken> GetOAuthToken() const;
  std::string GetDisplayName() const { return mUserInfo.displayName; }
  uint32_t GetUserId() const { return mUserId; }

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

 private:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  UserInfo mUserInfo;
  std::shared_ptr<OAuthToken> mOAuthToken;
  std::weak_ptr<UserRepository> mUserRepository;
  EventSource<IUserListener> mListeners;
  std::shared_ptr<ComponentContainer> mComponentContainer;

  UserId mUserId;
};
