/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/user.h"

#include <map>
#include <memory>

namespace ttv {
class TaskRunner;
class User;
class UserRepository;
class GetUserTask;
}  // namespace ttv

class ttv::UserRepository : public Component {
 public:
  using UserInfoLookupCallback = std::function<void(const ErrorDetails& ec, const UserInfo& userInfo)>;

 public:
  UserRepository();

  /**
   * Returns the anonymous user for the system.
   */
  std::shared_ptr<User> GetAnonymousUser() const { return mAnonymousUser; }

  /**
   * Returns the existing user or creates a new one if not already registered.
   */
  std::shared_ptr<User> RegisterUser(UserId userId);

  /**
   * Returns the existing user if registered.  This will NOT return the anonymous user.
   */
  std::shared_ptr<User> GetUser(UserId userId);

  /**
   * Fetches all active users.  This does NOT include the anonymous user.
   */
  TTV_ErrorCode GetUsers(std::vector<std::shared_ptr<User>>& users);

  /**
   * Removes the user from the registry.
   */
  TTV_ErrorCode UnRegisterUser(UserId userId);

  /**
   * Fetches user info by id.
   */
  TTV_ErrorCode FetchUserInfoById(UserId userId, UserInfoLookupCallback callback);

  /**
   * Fetches user info by name.
   */
  TTV_ErrorCode FetchUserInfoByName(const std::string& username, UserInfoLookupCallback callback);

  /**
   * Fetches user's own info using an auth token.
   */
  TTV_ErrorCode FetchUserInfoByAuthToken(const std::shared_ptr<OAuthToken>& authToken, UserInfoLookupCallback callback);

  /**
   * Returns cached user info matching the passed user id or TTV_EC_USERINFO_NOT_AVAILABLE if not yet fetched.  This
   * will NOT return info for the anonymous user.
   */
  TTV_ErrorCode GetUserInfoById(UserId userId, UserInfo& result);

  /**
   * Returns cached user info matching the passed username or TTV_EC_USERINFO_NOT_AVAILABLE if not yet fetched.  This
   * will NOT return info for the anonymous user.
   */
  TTV_ErrorCode GetUserInfoByName(const std::string& username, UserInfo& result);

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

 private:
  struct UserInfoLookupEntry {
    std::string username;
    std::shared_ptr<OAuthToken> authToken;
    UserId userId;
    UserInfoLookupCallback callback;
    std::shared_ptr<GetUserTask> task;
    bool forceRefetch;
  };

  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  TTV_ErrorCode ScheduleUserInfoLookup();
  void CompleteUserInfoLookup(const UserInfo& userInfo, const ErrorDetails& ec);

  TTV_ErrorCode CacheUserInfo(const UserInfo& userInfo);

  std::map<UserId, std::shared_ptr<User>> mUsers;
  std::vector<std::shared_ptr<User>> mCleanupUsers;  //!< The users that are shutting down.
  std::map<UserId, UserInfo> mUserInfo;              //!< A mapping of username to UserInfo.
  std::vector<UserInfoLookupEntry> mPendingUserInfoLookups;
  std::shared_ptr<User> mAnonymousUser;

  bool mLookupInProgress;
};
