/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/assertion.h"

#include <type_traits>

#include <cinttypes>
#include <tuple>

namespace ttv {
template <typename... VariantTypes>
class Variant;
}

/**
 * // An object that acts as a type-safe union; that is to say, it contains any one of the passed subtypes at any time.
 * // This object takes up as much space as an instance of the largest subtype plus one byte.
 * // @tparam Subtypes... The list of types that can be contained in the variant. A subtype may not be const, volatile,
 * //                     or a reference, and may not appear more than once in the list.
 * template <typename... Subtypes>
 * class ttv::Variant
 * {
 * public:
 *     // Constructs the variant with one of the variant's subtypes.
 *     template <typename Arg>
 *     Variant(Arg&& arg);
 *
 *     // Copy constructor (from another variant)
 *     Variant(const Variant<VariantTypes...>& src);
 *
 *     // Copy assignment (from another variant)
 *     Variant<VariantTypes...>& operator=(const Variant<VariantTypes...>& src);
 *
 *     // Move constructor (from another variant)
 *     Variant(Variant<VariantTypes...>&& src);
 *
 *     // Move assignment (from another variant)
 *     Variant<VariantTypes...>& operator=(Variant<VariantTypes...>&& src);
 *
 *     // Destructor
 *     ~Variant();
 *
 *     // Assign to this variant from one of its subtypes.
 *     template <typename Arg>
 *     Variant<VariantTypes...>& operator=(Arg&& arg);
 *
 *     // Returns whether variant contains the passed type.
 *     template <typename CheckType>
 *     bool Is();
 *
 *     // Returns a reference to the object as this type. It is illegal to call this method with the type that is not
 * contained
 *     // within the variant. Therefore, it is the responsibility of the client to call `Is` to verify the contained
 * type before
 *     // invoking this method.
 *     template <typename CastType>
 *     CastType& As();
 * };
 *
 * Example usage:
 *
 * std::variant<Foo, Bar> myVariant{Foo{}};
 * if (shouldReplaceWithBar)
 * {
 *     myVariant = Bar{};
 * }
 *
 * if (myVariant.Is<Foo>())
 * {
 *     Foo& myFoo = mVariant.As<Foo>();
 *     HandleFoo(myFoo);
 * }
 * else if (myVariant.Is<Bar>())
 * {
 *     Bar& myBar = mVariant.As<Bar>();
 *     HandleBar(myBar);
 * }
 *
 */

/**
 * ttv::Variant implementation.
 */
namespace ttv {
template <typename Element, typename... List>
struct TypeListContains;

template <typename Element, typename First, typename... Remaining>
struct TypeListContains<Element, First, Remaining...> {
  static constexpr bool Value = std::is_same<Element, First>::value || TypeListContains<Element, Remaining...>::Value;
};

template <typename Element>
struct TypeListContains<Element> {
  static constexpr bool Value = false;
};

template <typename First, typename... Remaining>
struct CheckVariantArguments {
  static_assert(!std::is_reference<First>::value, "Variant subtypes cannot be references.");
  static_assert(!std::is_const<First>::value, "Variant subtypes cannot be constant.");
  static_assert(!std::is_volatile<First>::value, "Variant subtypes cannot be volatile.");
  static_assert(!TypeListContains<First, Remaining...>::Value, "Variant subtypes must be unique in the type list.");
  using Next = typename CheckVariantArguments<Remaining...>::Check;

  using Check = void;
};

template <typename First>
struct CheckVariantArguments<First> {
  // Terminate recursion
  using Check = void;
};

template <typename ValueType>
std::enable_if_t<std::is_destructible<ValueType>::value> DestroyValue(ValueType& value) {
  value.~ValueType();
}

template <typename ValueType>
std::enable_if_t<!std::is_destructible<ValueType>::value> DestroyValue(ValueType& /*value*/) {}

template <uint8_t AtomIndex, typename First, typename... Remaining>
class VariantAtom;

template <uint8_t AtomIndex, typename Last>
class VariantAtom<AtomIndex, Last>;

template <uint8_t AtomIndex, typename First, typename... Remaining>
class VariantAtom {
 public:
  template <typename Arg>
  VariantAtom(Arg&& arg, std::enable_if_t<std::is_same<std::decay_t<Arg>, First>::value, int> /*enable*/ = 0) {
    new (&mValue.value) First{std::forward<Arg>(arg)};
  }

  template <typename Arg>
  VariantAtom(Arg&& arg, std::enable_if_t<!std::is_same<std::decay_t<Arg>, First>::value, int> /*enable*/ = 0) {
    new (&mValue.rest) VariantAtom<AtomIndex + 1, Remaining...>{std::forward<Arg>(arg)};
  }

  template <typename Arg>
  std::enable_if_t<std::is_same<std::decay_t<Arg>, First>::value> SetValue(Arg&& value) {
    new (&mValue.value) First{std::forward<Arg>(value)};
  }

  template <typename Arg>
  std::enable_if_t<!std::is_same<std::decay_t<Arg>, First>::value> SetValue(Arg&& value) {
    mValue.rest.SetValue(std::forward<Arg>(value));
  }

  void SetAtIndex(const VariantAtom& src, uint8_t activeIndex) {
    if (activeIndex == AtomIndex) {
      new (&mValue.value) First{src.mValue.value};
    } else {
      mValue.rest.SetAtIndex(src.mValue.rest, activeIndex);
    }
  }

  void SetAtIndex(VariantAtom&& src, uint8_t activeIndex) {
    if (activeIndex == AtomIndex) {
      new (&mValue.value) First{std::move(src.mValue.value)};
    } else {
      mValue.rest.SetAtIndex(std::move(src.mValue.rest), activeIndex);
    }
  }

  VariantAtom(const VariantAtom& src, uint8_t activeIndex) { SetAtIndex(src, activeIndex); }

  VariantAtom(VariantAtom&& src, uint8_t activeIndex) { SetAtIndex(std::move(src), activeIndex); }

  template <typename CheckType>
  uint8_t IndexOf() const {
    return std::is_same<CheckType, First>::value ? AtomIndex : mValue.rest.template IndexOf<CheckType>();
  }

  template <typename CastType>
  std::enable_if_t<std::is_same<First, CastType>::value, First&> As() {
    return mValue.value;
  }

  template <typename CastType>
  std::enable_if_t<!std::is_same<First, CastType>::value, CastType&> As() {
    return mValue.rest.template As<CastType>();
  }

  template <typename CastType>
  std::enable_if_t<std::is_same<First, CastType>::value, const First&> As() const {
    return mValue.value;
  }

  template <typename CastType>
  std::enable_if_t<!std::is_same<First, CastType>::value, const CastType&> As() const {
    return mValue.rest.template As<CastType>();
  }

  void DestroyAtIndex(uint8_t destroyIndex) {
    if (destroyIndex == AtomIndex) {
      DestroyValue(mValue.value);
    } else {
      mValue.rest.DestroyAtIndex(destroyIndex);
    }
  }

 private:
  union Value {
    Value() {}
    ~Value() {}

    First value;
    VariantAtom<AtomIndex + 1, Remaining...> rest;
  };

  Value mValue;
};

template <uint8_t AtomIndex, typename Last>
class VariantAtom<AtomIndex, Last> {
 public:
  template <typename Arg>
  VariantAtom(Arg&& arg, std::enable_if_t<std::is_same<std::decay_t<Arg>, Last>::value, int> /*enable*/ = 0) {
    new (&mValue.value) Last{std::forward<Arg>(arg)};
  }

  template <typename Arg>
  std::enable_if_t<std::is_same<std::decay_t<Arg>, Last>::value> SetValue(Arg&& value) {
    new (&mValue.value) Last{std::forward<Arg>(value)};
  }

  void SetAtIndex(const VariantAtom& src, uint8_t activeIndex) {
    TTV_ASSERT(activeIndex == AtomIndex);
    new (&mValue.value) Last{src.mValue.value};
  }

  void SetAtIndex(VariantAtom&& src, uint8_t activeIndex) {
    TTV_ASSERT(activeIndex == AtomIndex);
    new (&mValue.value) Last{std::move(src.mValue.value)};
  }

  VariantAtom(const VariantAtom& src, uint8_t activeIndex) { SetAtIndex(src, activeIndex); }

  VariantAtom(VariantAtom&& src, uint8_t activeIndex) { SetAtIndex(std::move(src), activeIndex); }

  template <typename CheckType>
  uint8_t IndexOf() const {
    return std::is_same<CheckType, Last>::value ? AtomIndex : AtomIndex + 1;
  }

  template <typename CastType>
  std::enable_if_t<std::is_same<Last, CastType>::value, Last&> As() {
    return mValue.value;
  }

  template <typename CastType>
  std::enable_if_t<std::is_same<Last, CastType>::value, const Last&> As() const {
    return mValue.value;
  }

  void DestroyAtIndex(uint8_t destroyIndex) {
    if (destroyIndex == AtomIndex) {
      DestroyValue(mValue.value);
    }
  }

 private:
  // Even though this only contains a single value, we put this into a union to ensure that we have manual control over
  // when the constructors/destructors are called for the value.
  union Value {
    Value() {}
    ~Value() {}

    Last value;
  };

  Value mValue;
};

template <typename... Subtypes>
class Variant {
 private:
  static_assert(sizeof...(Subtypes) > 0, "Cannot create a variant with an empty type list.");
  static_assert(sizeof...(Subtypes) < 256, "Variant cannot contain more than 255 types.");
  using CheckArgs = typename CheckVariantArguments<Subtypes...>::Check;
  using DefaultType = typename std::tuple_element_t<0, std::tuple<Subtypes...>>;

 public:
  Variant() : mAtom(DefaultType{}), mActiveTypeIndex(mAtom.template IndexOf<DefaultType>()) {}

  template <typename Arg, typename = std::enable_if_t<TypeListContains<std::decay_t<Arg>, Subtypes...>::Value>>
  Variant(Arg&& arg) : mAtom(std::forward<Arg>(arg)), mActiveTypeIndex(mAtom.template IndexOf<std::decay_t<Arg>>()) {}

  Variant(const Variant<Subtypes...>& src)
      : mAtom(src.mAtom, src.mActiveTypeIndex), mActiveTypeIndex(src.mActiveTypeIndex) {}

  Variant<Subtypes...>& operator=(const Variant<Subtypes...>& src) {
    mAtom.DestroyAtIndex(mActiveTypeIndex);
    mAtom.SetAtIndex(src.mAtom, src.mActiveTypeIndex);
    mActiveTypeIndex = src.mActiveTypeIndex;

    return *this;
  }

  Variant(Variant<Subtypes...>&& src)
      : mAtom(std::move(src.mAtom), src.mActiveTypeIndex), mActiveTypeIndex(src.mActiveTypeIndex) {}

  Variant<Subtypes...>& operator=(Variant<Subtypes...>&& src) {
    mAtom.DestroyAtIndex(mActiveTypeIndex);
    mAtom.SetAtIndex(std::move(src.mAtom), src.mActiveTypeIndex);
    mActiveTypeIndex = src.mActiveTypeIndex;

    return *this;
  }

  ~Variant() { mAtom.DestroyAtIndex(mActiveTypeIndex); }

  template <typename Arg, typename = std::enable_if_t<TypeListContains<std::decay_t<Arg>, Subtypes...>::Value>>
  Variant<Subtypes...>& operator=(Arg&& arg) {
    mAtom.DestroyAtIndex(mActiveTypeIndex);
    mAtom.SetValue(std::forward<Arg>(arg));
    mActiveTypeIndex = mAtom.template IndexOf<std::decay_t<Arg>>();

    return *this;
  }

  template <typename CheckType, typename = std::enable_if_t<TypeListContains<CheckType, Subtypes...>::Value>>
  bool Is() const {
    return (mAtom.template IndexOf<CheckType>() == mActiveTypeIndex);
  }

  template <typename CastType, typename = std::enable_if_t<TypeListContains<CastType, Subtypes...>::Value>>
  CastType& As() {
    TTV_ASSERT(Is<CastType>());
    return mAtom.template As<CastType>();
  }

  template <typename CastType, typename = std::enable_if_t<TypeListContains<CastType, Subtypes...>::Value>>
  const CastType& As() const {
    TTV_ASSERT(Is<CastType>());
    return mAtom.template As<CastType>();
  }

 private:
  VariantAtom<0, Subtypes...> mAtom;
  uint8_t mActiveTypeIndex;
};
}  // namespace ttv
