#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/core/channel/channelstatus.h"
#include "twitchsdk/core/channel/ichannellistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/igenericsubscriberlistener.h"
#include "twitchsdk/core/igenericsubscriberstatus.h"
#include "twitchsdk/core/profileimagestatus.h"

#include <fstream>
#include <streambuf>

using namespace ttv;

namespace ttv {
std::string ToString(PubSubState state) {
  switch (state) {
    case PubSubState::Disconnected:
      return "DISCONNECTED";
    case PubSubState::Connecting:
      return "CONNECTING";
    case PubSubState::Connected:
      return "CONNECTED";
    case PubSubState::Disconnecting:
      return "DISCONNECTING";
    default:
      return "???";
  }
}
}  // namespace ttv

namespace {
namespace ttv {
class ChannelStatus;
}

const char* kTraceComponents[] = {
  "CoreAPI",

  //"Socket",
  //"WebSocket",

  "TaskRunner",
  "User",
  "UserRepository",
  "PubSubClient",
  "PubSubClientConnection",
  "EventTracker",
};

class CoreApiListener;
class ChannelListener;

std::shared_ptr<Context> gContext;  // Cached by the static initializer
std::shared_ptr<CoreAPI> gCoreApi;
std::shared_ptr<CoreApiListener> gCoreApiListener;
std::map<std::pair<UserId, ChannelId>, std::shared_ptr<IChannelStatus>> gChannelStatusMap;
std::map<std::pair<UserId, std::string>, std::shared_ptr<IGenericSubscriberStatus>> gGenericSubscriberStatusMap;

class CoreApiListener : public ICoreAPIListener {
 public:
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) override {
    std::cout << "CoreApiListener::ModuleStateChanged: " << ::ToString(state) << " " << ErrorToString(ec) << std::endl;
  }

  virtual void CoreUserLoginComplete(
    const std::string& oauthToken, UserId userId, const ErrorDetails& errorDetails) override {
    std::cout << "CoreUserLoginComplete: " << userId << " " << oauthToken << " " << ErrorToString(errorDetails.ec)
              << std::endl;

    if (TTV_SUCCEEDED(errorDetails.ec)) {
      StaticInitializationRegistrar::GetInstance().RunUserLogin(gContext, userId);
    }
  }

  virtual void CoreUserLogoutComplete(UserId userId, TTV_ErrorCode ec) override {
    std::cout << "CoreUserLogoutComplete: " << userId << " " << ErrorToString(ec) << std::endl;

    if (TTV_SUCCEEDED(ec)) {
      StaticInitializationRegistrar::GetInstance().RunUserLogout(gContext, userId);
    }
  }

  virtual void CoreUserAuthenticationIssue(
    UserId userId, const std::string& oauthToken, const ErrorDetails& errorDetails) override {
    std::cout << "CoreUserAuthenticationIssue: " << userId << " " << oauthToken << " " << ErrorToString(errorDetails.ec)
              << std::endl;
  }

  virtual void CorePubSubStateChanged(UserId userId, PubSubState state, TTV_ErrorCode ec) override {
    std::cout << "CorePubSubStateChanged: " << userId << " " << ToString(state) << " " << ErrorToString(ec)
              << std::endl;
  }
};

class ChannelListener : public IChannelListener {
 public:
  ChannelListener(UserId userId, ChannelId channelId) : mUserId(userId), mChannelId(channelId) {}

  virtual void StreamUp(uint32_t playDelaySeconds) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] StreamUp with delay of "
              << playDelaySeconds << "seconds." << std::endl;
  }

  virtual void StreamDown() override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] StreamDown" << std::endl;
  }

  virtual void StreamViewerCountChanged(uint32_t viewerCount) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] Viewer Count changed to "
              << viewerCount << std::endl;
  }

  virtual void StreamTriggeredMidroll(uint32_t durationSeconds) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] StreamTriggeredMidroll "
              << durationSeconds << std::endl;
  }

  virtual void StreamReceivedWatchPartyUpdate(const WatchPartyUpdate& update) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId
              << "] StreamReceivedWatchPartyUpdate:" << std::endl;
    Print(update, " ");
  }

  virtual void ProfileImageUpdated(const std::vector<ProfileImage>& images) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] Profile Image Updated"
              << std::endl;
    std::cout << "Profile Images:" << std::endl;

    for (const auto& image : images) {
      Print(image, "  ");
    }
  }

  virtual void StreamInfoUpdated(StreamInfoUpdate&& info) override {
    std::cout << "ChannelListener[UserId:" << mUserId << " ChannelId:" << mChannelId << "] Stream Info Updated"
              << std::endl;
    Print(info, "  ");
  }

  virtual void SquadUpdated(SquadInfo&& squad) override {
    std::cout << "SquadUpdated: " << mUserId << " " << mChannelId << std::endl;
    Print(std::move(squad), "  ");
  }

  virtual void SquadLeft() override { std::cout << "SquadLeft: " << mUserId << " " << mChannelId << std::endl; }

  virtual void PixelTrackingUpdate(bool refresh) override {
    std::cout << "PixelTrackingUpdate: " << mUserId << " " << mChannelId << " " << refresh << std::endl;
  }

 private:
  const UserId mUserId;
  const ChannelId mChannelId;
};

class GenericSubscriberListener : public IGenericSubscriberListener {
 public:
  GenericSubscriberListener(UserId userId, const std::string& topic) : mUserId(userId), mTopic(topic) {}

  virtual void EventTopicData(const std::string& data) override {
    std::cout << "EventTopicData: " << mUserId << " " << mTopic << " " << data << std::endl;
  }

 private:
  UserId mUserId;
  std::string mTopic;
};

std::shared_ptr<CommandCategory> RegisterCoreCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Core", "The ttv::CoreAPI command interface");

  category->AddCommand("SetClientId")
    .AddFunction()
    .Description("Calls ttv::SetClientId")
    .AddParam("client-id", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = SetClientId(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CoreInit")
    .AddFunction()
    .Description("Calls CoreAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      gCoreApi->SetListener(gCoreApiListener);

      TTV_ErrorCode ec = gCoreApi->Initialize([](TTV_ErrorCode ec) {
        std::cout << "CoreInit completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CoreGetState")
    .AddFunction()
    .Description("Calls CoreAPI::GetState")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      IModule::State state = gCoreApi->GetState();
      std::cout << "  " << ::ToString(state) << std::endl;
      return true;
    });

  category->AddCommand("CoreShutdown")
    .AddFunction()
    .Description("Calls CoreAPI::Shutdown")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gCoreApi->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "CoreShutdown completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SetLocalLanguage")
    .AddFunction()
    .Description("Calls CoreAPI::SetLocalLanguage")
    .AddParam("language", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->SetLocalLanguage(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("GetLocalLanguage")
    .AddFunction()
    .Description("Calls CoreAPI::GetLocalLanguage")
    .AddParam("language", ParamType::String)
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::string language;
      TTV_ErrorCode ec = gCoreApi->GetLocalLanguage(language);
      ReportCommandResult(ec);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  Local language: " << language << std::endl;
      }
      return true;
    });

  category->AddCommand("Login")
    .AddFunction()
    .Description("Calls CoreAPI::Login")
    .AddParam("oauth", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->LogIn(params[0], [](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
        std::cout << "Login completed with error:";
        ReportCommandResult(errorDetails.ec);
        if (TTV_SUCCEEDED(errorDetails.ec)) {
          std::cout << "  UserInfo:" << std::endl;
          Print(userInfo, "    ");
        }
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("Logout")
    .AddFunction()
    .Description("Calls CoreAPI::LogOut")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->LogOut(params[0], [](TTV_ErrorCode ec) {
        std::cout << "Logout completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ConnectPubSub")
    .AddFunction()
    .Description("Calls CoreAPI::ConnectPubSub")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->ConnectPubSub(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("DisconnectPubSub")
    .AddFunction()
    .Description("Calls CoreAPI::DisconnectPubSub")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->DisconnectPubSub(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("FetchUserInfoById")
    .AddFunction()
    .Description("Calls CoreAPI::FetchUserInfoById")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gCoreApi->FetchUserInfoById(params[0], [](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
          std::cout << "FetchUserInfoById completed with error:";
          ReportCommandResult(errorDetails.ec);
          if (TTV_SUCCEEDED(errorDetails.ec)) {
            std::cout << "  UserInfo:" << std::endl;
            Print(userInfo, "    ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("FetchUserInfoByName")
    .AddFunction()
    .Description("Calls CoreAPI::FetchUserInfoByName")
    .AddParam("username", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gCoreApi->FetchUserInfoByName(params[0], [](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
          std::cout << "FetchUserInfoByName completed with error:";
          ReportCommandResult(errorDetails.ec);
          if (TTV_SUCCEEDED(errorDetails.ec)) {
            std::cout << "  UserInfo:" << std::endl;
            Print(userInfo, "    ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("FetchChannelInfoByName")
    .AddFunction()
    .Description("Calls CoreAPI::FetchChannelInfoByName")
    .AddParam("channelName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gCoreApi->FetchChannelInfoByName(params[0], [](TTV_ErrorCode ec, const ChannelInfo& channelInfo) {
          std::cout << "FetchChannelInfoByName completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  ChannelInfo:" << std::endl;
            Print(channelInfo, "    ");
          }
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("FetchChannelInfoById")
    .AddFunction()
    .Description("Calls CoreAPI::FetchChannelInfoById")
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gCoreApi->FetchChannelInfoById(params[0], [](TTV_ErrorCode ec, const ChannelInfo& channelInfo) {
          std::cout << "FetchChannelInfoById completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  ChannelInfo:" << std::endl;
            Print(channelInfo, "    ");
          }
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("FetchStreamInfoById")
    .AddFunction()
    .Description("Calls CoreAPI::FetchStreamInfoById")
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gCoreApi->FetchStreamInfoById(params[0], [](TTV_ErrorCode ec, const StreamInfo& streamInfo) {
        std::cout << "FetchStreamInfoById completed with error:";
        ReportCommandResult(ec);
        if (TTV_SUCCEEDED(ec)) {
          std::cout << "  StreamInfo:" << std::endl;
          Print(streamInfo, "    ");
        } else if (ec == TTV_EC_WEBAPI_RESULT_NO_STREAMINFO) {
          std::cout << "  StreamInfo Not Available" << std::endl;
        }
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateChannelStatus")
    .AddFunction()
    .Description("Calls CoreAPI::CreateChannelStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelStatusMap.find(pair);
      if (iter != gChannelStatusMap.end()) {
        std::cout << "Already created ChannelStatus for UserId/ChannelId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IChannelListener> listener = std::make_shared<ChannelListener>(pair.first, pair.second);
      std::shared_ptr<IChannelStatus> channelStatus;

      TTV_ErrorCode ec = gCoreApi->CreateChannelStatus(pair.first, pair.second, listener, channelStatus);
      gChannelStatusMap.insert({pair, channelStatus});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChannelStatus")
    .AddFunction()
    .Description("Calls CoreAPI::DisposeChannelStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelStatusMap.find(pair);
      if (iter == gChannelStatusMap.end()) {
        std::cout << "DisposeChannelStatus called on a ChannelStatus with UserId/ChannelId pair that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<IChannelStatus> channelStatus = iter->second;
      TTV_ErrorCode ec = channelStatus->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChannelStatusMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateGenericSubscriberStatus")
    .AddFunction()
    .Description("Calls CoreAPI::CreateGenericSubscriberStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("topic", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gGenericSubscriberStatusMap.find(pair);
      if (iter != gGenericSubscriberStatusMap.end()) {
        std::cout << "Already created GenericSubscriberStatus for UserId/Topic pair." << std::endl;
        return true;
      }

      std::shared_ptr<IGenericSubscriberListener> listener =
        std::make_shared<GenericSubscriberListener>(pair.first, pair.second);
      std::shared_ptr<IGenericSubscriberStatus> status;

      TTV_ErrorCode ec = gCoreApi->CreateGenericSubscriberStatus(pair.first, pair.second, listener, status);
      gGenericSubscriberStatusMap.insert({pair, status});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeGenericSubscriberStatus")
    .AddFunction()
    .Description("Calls CoreAPI::DisposeGenericSubscriberStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("genericSubscriberId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gGenericSubscriberStatusMap.find(pair);
      if (iter == gGenericSubscriberStatusMap.end()) {
        std::cout << "DisposeGenericSubscriberStatus called on a status with UserId " << pair.first << " Topic "
                  << pair.second << " that doesn't exist." << std::endl;
        return true;
      }

      std::shared_ptr<IGenericSubscriberStatus> status = iter->second;
      TTV_ErrorCode ec = status->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gGenericSubscriberStatusMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("UploadProfileImage")
    .AddFunction()
    .Description("Calls ProfileImageStatus::UploadProfileImage")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("filePath", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      std::pair<UserId, ChannelId> pair = {userId, params[1].GetUInt32()};

      auto iter = gChannelStatusMap.find(pair);
      if (iter == gChannelStatusMap.end()) {
        std::cout << "UploadProfileImage called on a ChannelStatus with UserId/ChannelId pair that doesn't exist."
                  << std::endl;
        return true;
      }

      std::ifstream response(params[2], std::ios::binary);
      const bool opened = response.is_open();
      if (!opened) {
        std::cout << "UploadProfileImage: Could not open file \"" << std::string(params[2]) << "\"" << std::endl;
        return true;
      }

      std::string data((std::istreambuf_iterator<char>(response)), std::istreambuf_iterator<char>());
      TTV_ErrorCode ec = iter->second->UploadProfileImage(
        data.c_str(), data.size(), [userId](TTV_ErrorCode ec, const std::vector<ProfileImage>& images) {
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "Profile Image for UserId:" << userId << " Upload Succeeded" << std::endl;
            std::cout << "Profile Images:" << std::endl;

            for (const auto& image : images) {
              Print(image, "  ");
            }
          } else {
            std::cout << "Profile Image for UserId:" << userId << " Upload Failed with ErrorCode: " << ErrorToString(ec)
                      << std::endl;
          }
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("SetGlobalSetting")
    .AddFunction()
    .Description("Calls CoreAPI::SetGlobalSetting")
    .AddParam("key", ParamType::String)
    .AddParam("value", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = TrimStringQuotes(params[0]);
      std::string value = TrimStringQuotes(params[1]);

      gCoreApi->SetGlobalSetting(key, value);

      std::cout << "  '" << key << "' = '" << value << "'" << std::endl;

      return true;
    });

  category->AddCommand("GetGlobalSetting")
    .AddFunction()
    .Description("Calls CoreAPI::SetGlobalSetting")
    .AddParam("key", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = TrimStringQuotes(params[0]);
      std::string value;

      TTV_ErrorCode ec = gCoreApi->GetGlobalSetting(key, value);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  '" << key << "' = '" << value << "'" << std::endl;
      } else {
        std::cout << "  Setting not found: '" << key << "'" << std::endl;
      }

      return true;
    });

  category->AddCommand("RemoveGlobalSetting")
    .AddFunction()
    .Description("Calls CoreAPI::RemoveGlobalSetting")
    .AddParam("key", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = TrimStringQuotes(params[0]);

      TTV_ErrorCode ec = gCoreApi->RemoveGlobalSetting(key);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  Removed setting: '" << key << "'" << std::endl;
      } else {
        std::cout << "  Setting not found: '" << key << "'" << std::endl;
      }

      return true;
    });

  category->AddCommand("PrintErrorCodes")
    .AddFunction()
    .Description("Prints out a table of all registered TTV_ErrorCode values.")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::vector<EnumValue> values;
      GetAllErrorCodes(values);

      for (const auto& v : values) {
        std::cout << v.name << "\t0x" << std::hex << v.value << "\t" << std::dec << v.value << std::endl;
      }

      return true;
    });

  category->AddCommand("PrintJavaCoreErrorCodes")
    .AddFunction()
    .Description("Dumps the Java snippet for updated values in CoreErrorCode.java")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::vector<EnumValue> values;
      GetCoreErrorCodeValues(values);

      for (const auto& v : values) {
        std::cout << "public static final ErrorCode " << v.name << " = new CoreErrorCode(0x" << std::hex << v.value
                  << ", \"" << v.name << "\");" << std::endl;
      }

      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      gContext = context;

      gCoreApi = std::make_shared<CoreAPI>();
      context->RegisterModule(gCoreApi);

      gCoreApiListener = std::make_shared<CoreApiListener>();

      auto commands = RegisterCoreCommands(context);
      context->RegisterCommandCategory(commands);
    },
    0);
});
}  // namespace
