/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/channel/channelrepository.h"

#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getchanneltask.h"
#include "twitchsdk/core/task/taskrunner.h"

namespace {
const uint64_t kMaxCacheUnusedDuration = 1000 * 60 * 60;
const char* kLoggerName = "ChannelRepository";
}  // namespace

ttv::ChannelRepository::ChannelRepository() : mLookupInProgress(false) {
  TTV_ErrorCode ret = CreateMutex(mMutex, "ChannelRepository");
  ASSERT_ON_ERROR(ret);
}

std::string ttv::ChannelRepository::GetLoggerName() const {
  return kLoggerName;
}

void ttv::ChannelRepository::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  ScheduleLookup();

  {
    AutoMutex lock(mMutex.get());

    // Clean out the cache
    mChannelInfoCache.PurgeUnused(kMaxCacheUnusedDuration);
    mChannelInfoCache.PurgeExpired();
  }

  Component::Update();
}

TTV_ErrorCode ttv::ChannelRepository::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    AutoMutex lock(mMutex.get());
    // Abort pending lookups
    for (auto iter = mPendingChannelInfoLookups.begin(); iter != mPendingChannelInfoLookups.end();) {
      if (iter->task == nullptr) {
        if (iter->callback != nullptr) {
          iter->callback(TTV_EC_REQUEST_ABORTED, {});
        }
        iter = mPendingChannelInfoLookups.erase(iter);
      } else {
        ++iter;
      }
    }
  }

  return ec;
}

bool ttv::ChannelRepository::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  AutoMutex lock(mMutex.get());
  if (mLookupInProgress) {
    return false;
  }

  return true;
}

void ttv::ChannelRepository::CompleteShutdown() {
  Component::CompleteShutdown();

  AutoMutex lock(mMutex.get());
  TTV_ASSERT(!mLookupInProgress);

  mPendingChannelInfoLookups.clear();
  mChannelInfoCache.Clear();
}

TTV_ErrorCode ttv::ChannelRepository::FetchChannelInfo(ChannelId channelId, LookupCallback callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  LookupEntry entry;
  entry.channelId = channelId;
  entry.callback = callback;

  AutoMutex lock(mMutex.get());
  mPendingChannelInfoLookups.push_back(entry);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ChannelRepository::GetChannelInfo(ChannelId channelId, ChannelInfo& result) {
  AutoMutex lock(mMutex.get());
  if (mChannelInfoCache.GetEntry(channelId, result)) {
    mChannelInfoCache.MarkEntryUsed(channelId);
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_NOT_AVAILABLE;
  }
}

void ttv::ChannelRepository::CompleteLookup(const ChannelInfo& channelInfo, TTV_ErrorCode ec) {
  LookupCallback callback = nullptr;
  {
    AutoMutex lock(mMutex.get());
    mLookupInProgress = false;

    callback = mPendingChannelInfoLookups[0].callback;
    mPendingChannelInfoLookups.erase(mPendingChannelInfoLookups.begin());
  }

  if (callback != nullptr) {
    callback(ec, channelInfo);
  }
}

TTV_ErrorCode ttv::ChannelRepository::ScheduleLookup() {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  ChannelId channelId = 0;
  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  {
    AutoMutex lock(mMutex.get());
    if (mLookupInProgress) {
      return TTV_EC_REQUEST_PENDING;
    }

    if (mPendingChannelInfoLookups.empty()) {
      return TTV_EC_SUCCESS;
    }

    LookupEntry& entry = mPendingChannelInfoLookups[0];
    channelId = entry.channelId;

    GetChannelTask::Callback fetchCallback = [this](GetChannelTask* source, TTV_ErrorCode callbackEc,
                                               std::shared_ptr<GetChannelTask::Result> result) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(callbackEc)) {
        CacheChannelInfo(result->channelInfo);
        CompleteLookup(result->channelInfo, callbackEc);
      } else {
        CompleteLookup({}, callbackEc);
      }
    };

    entry.task = std::make_shared<GetChannelTask>(channelId, fetchCallback);
    ec = StartTask(entry.task);

    if (TTV_SUCCEEDED(ec)) {
      mLookupInProgress = true;
    }
  }

  if (TTV_FAILED(ec)) {
    CompleteLookup({}, TTV_EC_SHUTTING_DOWN);
  }

  return ec;
}

TTV_ErrorCode ttv::ChannelRepository::CacheChannelInfo(const ChannelInfo& channelInfo) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!IsValidChannelName(channelInfo.name) || channelInfo.channelId == 0) {
    return TTV_EC_INVALID_ARG;
  }

  AutoMutex lock(mMutex.get());
  mChannelInfoCache.SetEntry(channelInfo.channelId, channelInfo);

  return TTV_EC_SUCCESS;
}
