/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/channel/channelstatus.h"

#include "twitchsdk/core/channel/ichannellistener.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/profileimagestatus.h"

namespace {
const char* kLoggerName = "StreamStatus";
const char* kStreamUpdateTopicPrefix = "broadcast-settings-update.";
const char* kVideoStreamTopicPrefix = "video-playback-by-id.";
const char* kChannelSquadUpdatesTopicPrefix = "channel-squad-updates.";
const char* kAdPropertyRefreshPrefix = "ad-property-refresh.";
}  // namespace

ttv::ChannelStatus::ChannelStatus(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user),
      mStreamUpdatePubSubTopic(kStreamUpdateTopicPrefix + std::to_string(channelId)),
      mVideoStreamPubSubTopic(kVideoStreamTopicPrefix + std::to_string(channelId)),
      mChannelSquadUpdatesPubSubTopic(kChannelSquadUpdatesTopicPrefix + std::to_string(channelId)),
      mAdPropertyRefreshPubSubTopic(kAdPropertyRefreshPrefix + std::to_string(channelId)),
      mChannelId(channelId) {
  AddTopic(mStreamUpdatePubSubTopic);
  AddTopic(mVideoStreamPubSubTopic);
  AddTopic(mChannelSquadUpdatesPubSubTopic);
  AddTopic(mAdPropertyRefreshPubSubTopic);
}

// Component Overrides
TTV_ErrorCode ttv::ChannelStatus::Initialize() {
  TTV_ErrorCode ec = PubSubComponent::Initialize();
  if (TTV_SUCCEEDED(ec)) {
    auto user = mUser.lock();
    if (user != nullptr) {
      std::shared_ptr<ProfileImageStatus> profileImageStatus = std::make_shared<ProfileImageStatus>(user, mChannelId);
      profileImageStatus->SetListener(mListener);
      profileImageStatus->SetTaskRunner(mTaskRunner);
      ec = profileImageStatus->Initialize();
      TTV_ASSERT(TTV_SUCCEEDED(ec));

      mComponentContainer = std::make_shared<ComponentContainer>();

      ec = mComponentContainer->Initialize();
      TTV_ASSERT(TTV_SUCCEEDED(ec));

      ec = mComponentContainer->SetComponent(ProfileImageStatus::GetComponentName(), profileImageStatus);
      TTV_ASSERT(TTV_SUCCEEDED(ec));
    } else {
      return TTV_EC_NEED_TO_LOGIN;
    }
  }

  return ec;
}

bool ttv::ChannelStatus::CheckShutdown() {
  if (!PubSubComponent::CheckShutdown()) {
    return false;
  }

  if (mComponentContainer != nullptr) {
    if (mComponentContainer->GetState() == ComponentContainer::State::Initialized) {
      mComponentContainer->Shutdown();
    }

    if (mComponentContainer->GetState() != ComponentContainer::State::Uninitialized) {
      return false;
    }
  }

  return true;
}

void ttv::ChannelStatus::CompleteShutdown() {
  mComponentContainer.reset();

  PubSubComponent::CompleteShutdown();
}

void ttv::ChannelStatus::Update() {
  PubSubComponent::Update();

  if (mComponentContainer != nullptr) {
    mComponentContainer->Update();
  }
}

TTV_ErrorCode ttv::ChannelStatus::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ChannelStatus::UploadProfileImage(
  const char* imageBuffer, size_t imageSize, const UploadProfileImageCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  std::shared_ptr<ProfileImageStatus> profileImageStatus = mComponentContainer->GetComponent<ProfileImageStatus>();

  if (profileImageStatus == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return profileImageStatus->UploadProfileImage(imageBuffer, imageSize, callback);
}

std::string ttv::ChannelStatus::GetLoggerName() const {
  return kLoggerName;
}

void ttv::ChannelStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug,
    "ChannelListener (video-playback-by-id.) SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::ChannelStatus::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mVideoStreamPubSubTopic) {
    std::string type;

    if (!ParseString(msg, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    if (type == "viewcount") {
      const auto& jViewerCount = msg["viewers"];
      if (jViewerCount.isNumeric()) {
        uint32_t viewerCount = static_cast<uint32_t>(jViewerCount.asUInt());

        if (mListener != nullptr) {
          mListener->StreamViewerCountChanged(viewerCount);
        }
      }
    } else if (type == "stream-up") {
      const auto& jDelay = msg["play_delay"];
      uint32_t delay = jDelay.isNumeric() ? static_cast<uint32_t>(jDelay.asUInt()) : 0;

      if (mListener != nullptr) {
        mListener->StreamUp(delay);
      }
    } else if (type == "stream-down") {
      if (mListener != nullptr) {
        mListener->StreamDown();
      }
    } else if (type == "commercial") {
      uint32_t durationSeconds = 30;

      const auto& jLength = msg["length"];
      if (jLength.isNumeric()) {
        durationSeconds = static_cast<uint32_t>(jLength.asUInt());
      }

      if (mListener != nullptr) {
        mListener->StreamTriggeredMidroll(durationSeconds);
      }
    } else if (type == "watchparty-vod") {
      WatchPartyUpdate update;
      const auto& jVod = msg["vod"];
      if (!jVod.isNull() && jVod.isObject()) {
        const auto& jIncrementUrl = jVod["increment_url"];
        if (!jIncrementUrl.isNull() && jIncrementUrl.isString()) {
          update.incrementUrl = jIncrementUrl.asString();
        }

        const auto& jVodId = jVod["vod_id"];
        if (!jVodId.isNull() && jVodId.isString()) {
          update.vodId = jVodId.asString();
        }

        const auto& jTitle = jVod["title"];
        if (!jTitle.isNull() && jTitle.isString()) {
          update.title = jTitle.asString();
        }

        const auto& jWatchPartyId = jVod["wp_id"];
        if (!jWatchPartyId.isNull() && jWatchPartyId.isString()) {
          update.watchPartyId = jWatchPartyId.asString();
        }

        const auto& jBroadcastType = jVod["broadcast_type"];
        if (!jBroadcastType.isNull() && jBroadcastType.isString()) {
          std::string broadcastTypeString = jBroadcastType.asString();
          if (broadcastTypeString == "archive") {
            update.broadcastType = VodType::Archive;
          } else if (broadcastTypeString == "highlight") {
            update.broadcastType = VodType::Highlight;
          } else if (broadcastTypeString == "upload") {
            update.broadcastType = VodType::Upload;
          }
        }

        const auto& jViewable = jVod["viewable"];
        if (!jViewable.isNull() && jViewable.isString()) {
          if (jViewable == "public") {
            update.viewable = true;
          } else if (jViewable == "private") {
            update.viewable = false;
          }
        }

        if (mListener != nullptr) {
          mListener->StreamReceivedWatchPartyUpdate(update);
        }
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  } else if (topic == mStreamUpdatePubSubTopic) {
    StreamInfoUpdate info;
    if (!ttv::json::ToObject(msg, info)) {
      return;
    }

    if (mListener != nullptr) {
      mListener->StreamInfoUpdated(std::move(info));
    }
  } else if (topic == mChannelSquadUpdatesPubSubTopic) {
    const auto& jType = msg["type"];
    if (!jType.isString() || jType.asString() != "squad") {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message wrong type, dropping");
      return;
    }

    const auto& jSquad = msg["squad"];
    if (jSquad.isNull() || !jSquad.isObject()) {
      mListener->SquadLeft();
    } else {
      SquadInfo squad;
      if (!ttv::json::ToObject(jSquad, squad)) {
        return;
      }

      mListener->SquadUpdated(std::move(squad));
    }
  } else if (topic == mAdPropertyRefreshPubSubTopic) {
    std::string type;

    if (!ParseString(msg, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    if (type == "pixel_tracking_update") {
      bool result;
      auto jData = msg["data"];
      if (jData.isNull() || !ParseBool(jData, "refresh", result)) {
        Log(MessageLevel::Error, "Malformed packet %s", type.c_str());
        return;
      }

      mListener->PixelTrackingUpdate(result);
    }
  }
}
