/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/component.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"

#include <algorithm>

namespace {
const uint64_t kShutdownThresholdMilliseconds = 2000;
const char* kContainerLoggerName = "ComponentContainer";
}  // namespace

ttv::Component::Component() : mState(State::Uninitialized), mShutdownTimeMilliseconds(0) {
  TTV_ErrorCode ec = CreateMutex(mTaskMutex, "Component");
  TTV_ASSERT(ec == TTV_EC_SUCCESS);
}

ttv::Component::~Component() {
  if (mState != State::Uninitialized && mState != State::Inert) {
    ttv::trace::Message("Component", MessageLevel::Error, "~Component() - component not shutdown prior to destruction");
    TTV_ASSERT(false);
  }
}

void ttv::Component::SetTaskRunner(std::shared_ptr<TaskRunner> taskRunner) {
  mTaskRunner = taskRunner;
}

void ttv::Component::SetMainEventScheduler(const std::shared_ptr<IEventScheduler>& eventScheduler) {
  mMainEventScheduler = eventScheduler;
}

ttv::IComponent::State ttv::Component::GetState() const {
  return mState.client;
}

void ttv::Component::SetState(State state) {
  ttv::trace::Message(
    "Component", MessageLevel::Debug, "%s SetState: %u", GetLoggerName().c_str(), static_cast<uint32_t>(state));

  mState = state;
}

void ttv::Component::SetClientState(State state) {
  if (mState.client == state) {
    return;
  }

  ttv::trace::Message(
    "Component", MessageLevel::Debug, "%s SetClientState: %u", GetLoggerName().c_str(), static_cast<uint32_t>(state));

  mState.client = state;
}

void ttv::Component::SetServerState(State state) {
  if (mState.server == state) {
    return;
  }

  ttv::trace::Message(
    "Component", MessageLevel::Debug, "%s SetServerState: %u", GetLoggerName().c_str(), static_cast<uint32_t>(state));

  mState.server = state;
}

TTV_ErrorCode ttv::Component::Initialize() {
  switch (mState.client) {
    case State::Uninitialized: {
      mState = State::Initialized;
      return TTV_EC_SUCCESS;
    }
    case State::Initialized: {
      return TTV_EC_ALREADY_INITIALIZED;
    }
    case State::ShuttingDown: {
      return TTV_EC_SHUTTING_DOWN;
    }
    case State::Inert:
    default: { return TTV_EC_SUCCESS; }
  }
}

void ttv::Component::Update() {
  if (mState.client == State::ShuttingDown) {
    if (CheckShutdown()) {
      CompleteShutdown();
    }
  }
}

TTV_ErrorCode ttv::Component::Shutdown() {
  switch (mState.client) {
    case State::ShuttingDown:
      return TTV_EC_SHUTTING_DOWN;
    case State::Uninitialized:
      return TTV_EC_NOT_INITIALIZED;
    default:
      break;
  }

  ttv::trace::Message("Component", MessageLevel::Debug, "Shutdown() called on %s", GetLoggerName().c_str());
  mShutdownTimeMilliseconds = ttv::GetSystemTimeMilliseconds();

  SetClientState(State::ShuttingDown);

  {
    ttv::AutoMutex lock(mTaskMutex.get());
    for (const auto& task : mRunningTasks) {
      task->Abort();
    }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::Component::CheckShutdown() {
  if (ttv::GetSystemTimeMilliseconds() - mShutdownTimeMilliseconds > kShutdownThresholdMilliseconds) {
    ttv::trace::Message(
      "Component", MessageLevel::Error, "%s is taking a long time to shut down.", GetLoggerName().c_str());
  }

  ttv::AutoMutex lock(mTaskMutex.get());
  return mRunningTasks.empty();
}

void ttv::Component::CompleteShutdown() {
  SetState(State::Uninitialized);

  mTaskRunner.reset();
  mRunningTasks.clear();
}

TTV_ErrorCode ttv::Component::StartTask(std::shared_ptr<Task> task) {
  // Can't start tasks while shutting down
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  TTV_ASSERT(mTaskRunner != nullptr);

  {
    ttv::AutoMutex lock(mTaskMutex.get());
    mRunningTasks.emplace_back(task);
  }

  bool succeeded = mTaskRunner->AddTask(task);
  if (succeeded) {
    return TTV_EC_SUCCESS;
  } else {
    CompleteTask(task.get());

    return TTV_EC_SHUTTING_DOWN;
  }
}

void ttv::Component::CompleteTask(Task* task) {
  ttv::AutoMutex lock(mTaskMutex.get());
  auto iter = std::find_if(mRunningTasks.begin(), mRunningTasks.end(),
    [task](const std::shared_ptr<Task>& cur) -> bool { return static_cast<Task*>(cur.get()) == task; });

  TTV_ASSERT(iter != mRunningTasks.end());
  if (iter != mRunningTasks.end()) {
    mRunningTasks.erase(iter);
  }
}

bool ttv::Component::IsTaskRunning(Task* task) const {
  ttv::AutoMutex lock(mTaskMutex.get());
  auto iter = std::find_if(mRunningTasks.begin(), mRunningTasks.end(),
    [task](const std::shared_ptr<Task>& cur) -> bool { return static_cast<Task*>(cur.get()) == task; });

  return iter != mRunningTasks.end();
}

ttv::UserComponent::UserComponent(const std::shared_ptr<User>& user) : mUser(user), mOAuthIssue(false) {}

TTV_ErrorCode ttv::UserComponent::Initialize() {
  TTV_ErrorCode ec = Component::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    auto user = mUser.lock();
    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }

    // Subscribe for user events
    mUserListener = std::make_shared<UserListener>(this);
    user->AddListener(mUserListener);

    // Check OAuth token validity
    if (!user->GetOAuthToken()->GetValid()) {
      Log(MessageLevel::Debug, "User OAuth token invalid");

      mOAuthIssue = true;
    }
  }

  return ec;
}

std::shared_ptr<ttv::User> ttv::UserComponent::GetUser() {
  return mUser.lock();
}

void ttv::UserComponent::CompleteShutdown() {
  Component::CompleteShutdown();

  auto user = mUser.lock();
  if (user != nullptr && mUserListener != nullptr) {
    user->RemoveListener(mUserListener);
    mUserListener.reset();
  }
}

void ttv::UserComponent::Log(MessageLevel level, const char* format, ...) {
  auto logger = GetLoggerName();

  MessageLevel activeLevel = MessageLevel::None;
  ttv::trace::GetComponentMessageLevel(logger.c_str(), activeLevel);
  if (level < activeLevel) {
    return;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  // Append the username onto the message
  std::string str("[");
  str += user->GetUserName();
  str += "] ";
  str += format;

  va_list args;
  va_start(args, format);
  ttv::trace::MessageVaList(logger.c_str(), level, str.c_str(), args);
  va_end(args);
}

void ttv::UserComponent::OnUserLogInComplete(TTV_ErrorCode ec) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  mOAuthIssue = TTV_FAILED(ec);

  if (mOAuthIssue) {
    Log(MessageLevel::Debug, "Pausing internal updating due to OAuth issue");
  } else {
    Log(MessageLevel::Debug, "Allowing internal updating due to successful login");
  }
}

void ttv::UserComponent::OnUserLogOutComplete(TTV_ErrorCode /*ec*/) {}

void ttv::UserComponent::OnUserInfoFetchComplete(TTV_ErrorCode /*ec*/) {}

void ttv::UserComponent::OnUserAuthenticationIssue(
  std::shared_ptr<const OAuthToken> /*oauthToken*/, TTV_ErrorCode /*ec*/) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  mOAuthIssue = true;
}

ttv::UserComponent::UserListener::UserListener(UserComponent* owner) : mOwner(owner) {}

void ttv::UserComponent::UserListener::OnUserLogInComplete(User* /*source*/, TTV_ErrorCode ec) {
  mOwner->OnUserLogInComplete(ec);
}

void ttv::UserComponent::UserListener::OnUserLogOutComplete(User* /*source*/, TTV_ErrorCode ec) {
  mOwner->OnUserLogOutComplete(ec);
}

void ttv::UserComponent::UserListener::OnUserInfoFetchComplete(User* /*source*/, TTV_ErrorCode ec) {
  mOwner->OnUserInfoFetchComplete(ec);
}

void ttv::UserComponent::UserListener::OnUserAuthenticationIssue(
  User* /*source*/, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec) {
  mOwner->OnUserAuthenticationIssue(oauthToken, ec);
}

ttv::ComponentContainer::ComponentContainer() {}

ttv::ComponentContainer::~ComponentContainer() {
  TTV_ASSERT(mState == State::Uninitialized);
}

std::string ttv::ComponentContainer::GetLoggerName() const {
  return kContainerLoggerName;
}

TTV_ErrorCode ttv::ComponentContainer::Initialize() {
  TTV_ErrorCode ec = Component::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    TTV_ErrorCode ret = CreateMutex(mMutex, "ComponentContainer");
    ASSERT_ON_ERROR(ret);
  }

  return ec;
}

void ttv::ComponentContainer::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  {
    std::vector<std::shared_ptr<IComponent>> components;
    std::vector<std::shared_ptr<IComponent>> disposeComponents;

    // Create copies so we can update while unlocked
    {
      ttv::AutoMutex am(mMutex.get());

      components = mComponents;
      disposeComponents = mDisposeComponents;
    }

    // Update live components
    for (auto component : components) {
      component->Update();
    }

    // Update components that are shutting down
    for (auto iter = disposeComponents.begin(); iter != disposeComponents.end();) {
      auto component = *iter;

      component->Update();

      if (component->GetState() == Component::State::Uninitialized ||
          component->GetState() == Component::State::Inert) {
        ++iter;
      } else {
        iter = disposeComponents.erase(iter);
      }
    }

    // Release references to any components that are ready to be destroyed
    if (!disposeComponents.empty()) {
      ttv::AutoMutex am(mMutex.get());

      for (auto component : disposeComponents) {
        auto iter = std::find(mDisposeComponents.begin(), mDisposeComponents.end(), component);
        TTV_ASSERT(iter != mDisposeComponents.end());

        if (iter != mDisposeComponents.end()) {
          mDisposeComponents.erase(iter);
        }
      }
    }
  }

  Component::Update();
}

TTV_ErrorCode ttv::ComponentContainer::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    std::vector<std::shared_ptr<IComponent>> components;

    // Create a copy so we can shutdown unlocked
    {
      ttv::AutoMutex am(mMutex.get());

      components = mComponents;

      for (auto component : components) {
        mDisposeComponents.push_back(component);
      }

      mComponentMap.clear();
    }

    for (auto component : components) {
      component->Shutdown();
    }
  }

  return ec;
}

bool ttv::ComponentContainer::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  ttv::AutoMutex am(mMutex.get());

  if (!mDisposeComponents.empty()) {
    return false;
  }

  for (auto component : mComponents) {
    if (component->GetState() != IComponent::State::Uninitialized &&
        component->GetState() != IComponent::State::Inert) {
      return false;
    }
  }

  return true;
}

void ttv::ComponentContainer::CompleteShutdown() {
  Component::CompleteShutdown();

  {
    ttv::AutoMutex am(mMutex.get());

    mDisposeComponents.clear();
    mComponents.clear();
    mComponentMap.clear();
  }

  // TODO: Still need to use unique_ptr for automatic deletion?
  mMutex.reset();
}

TTV_ErrorCode ttv::ComponentContainer::AddComponent(std::shared_ptr<IComponent> component) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (component == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  ttv::AutoMutex am(mMutex.get());

  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  if (iter == mComponents.end()) {
    mComponents.push_back(component);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::RemoveComponent(std::shared_ptr<IComponent> component) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (component == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  ttv::AutoMutex am(mMutex.get());

  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  if (iter != mComponents.end()) {
    mComponents.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::DisposeComponent(std::shared_ptr<IComponent> component) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (component == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  {
    ttv::AutoMutex am(mMutex.get());

    auto iter = std::find(mComponents.begin(), mComponents.end(), component);
    if (iter != mComponents.end()) {
      mDisposeComponents.push_back(component);
      mComponents.erase(iter);
    } else {
      component.reset();
    }
  }

  if (component != nullptr) {
    component->Shutdown();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::SetComponent(const std::string& name, std::shared_ptr<IComponent> component) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (component == nullptr || name == "") {
    return TTV_EC_INVALID_ARG;
  }

  ttv::AutoMutex am(mMutex.get());

  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  if (iter == mComponents.end()) {
    mComponents.push_back(component);
  }

  mComponentMap[name] = component;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::GetComponent(const std::string& name, std::shared_ptr<IComponent>& result) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  ttv::AutoMutex am(mMutex.get());

  auto iter = mComponentMap.find(name);
  if (iter == mComponentMap.end()) {
    return TTV_EC_INVALID_ARG;
  }

  result = iter->second;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::RemoveComponent(const std::string& name) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  ttv::AutoMutex am(mMutex.get());

  auto mapIter = mComponentMap.find(name);
  if (mapIter == mComponentMap.end()) {
    return TTV_EC_INVALID_ARG;
  }

  auto component = mapIter->second;

  mComponentMap.erase(mapIter);

  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  if (iter != mComponents.end()) {
    mComponents.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ComponentContainer::DisposeComponent(const std::string& name) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  std::shared_ptr<IComponent> component;

  {
    ttv::AutoMutex am(mMutex.get());

    auto mapIter = mComponentMap.find(name);
    if (mapIter == mComponentMap.end()) {
      return TTV_EC_INVALID_ARG;
    }

    component = mapIter->second;

    auto iter = std::find(mComponents.begin(), mComponents.end(), component);
    if (iter != mComponents.end()) {
      mComponents.erase(iter);
    }

    mDisposeComponents.push_back(component);
    mComponentMap.erase(mapIter);
  }

  component->Shutdown();

  return TTV_EC_SUCCESS;
}

bool ttv::ComponentContainer::ContainsComponent(std::shared_ptr<IComponent> component) const {
  ttv::AutoMutex am(mMutex.get());

  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  return iter != mComponents.end();
}
