/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/coreapi.h"

#include "twitchsdk/core/channel/channelrepository.h"
#include "twitchsdk/core/channel/channelstatus.h"
#include "twitchsdk/core/dashboardactivitystatus.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/genericsubscriberstatus.h"
#include "twitchsdk/core/pollingeventscheduler.h"
#include "twitchsdk/core/profileimagestatus.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getstreamtask.h"
#include "twitchsdk/core/task/lambdatask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/task/validateoauthtask.h"
#include "twitchsdk/core/types/memorytypes.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

namespace {
const char* kModuleName = "ttv::CoreAPI";
const char* kRequiredScopes[] = {"user_read"};

char gTwitchClientId[128] = "";
}  // namespace

struct ttv::CoreAPIInternalData {
  std::unique_ptr<IMutex> mutex;  // The mutex that guards the instance containers here
  std::vector<std::shared_ptr<ChannelStatus>> channelStatusInstances;
  std::vector<std::shared_ptr<DashboardActivityStatus>> dashboardActivityStatusInstances;
  std::vector<std::shared_ptr<GenericSubscriberStatus>> genericSubscriberStatusInstances;
};

TTV_ErrorCode ttv::SetClientId(const std::string& clientId) {
  std::string clean = clientId;
  Trim(clean);

  // TODO: We could perhaps validate the client id somehow

  if (clean == "" || clean.size() >= sizeof(gTwitchClientId)) {
    return TTV_EC_INVALID_ARG;
  }

  memcpy(gTwitchClientId, clean.data(), clean.size());
  gTwitchClientId[clean.size()] = '\0';

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::GetClientId(std::string& clientId) {
  clientId = gTwitchClientId;

  return TTV_EC_SUCCESS;
}

ttv::ICoreApiClient::ICoreApiClient() {}

void ttv::ICoreApiClient::GetRequiredOAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

void ttv::ICoreApiClient::CoreUserLoggedIn(std::shared_ptr<User> /*user*/) {
  // Clients don't need to implement this
}

void ttv::ICoreApiClient::CoreUserLoggedOut(std::shared_ptr<User> /*user*/) {
  // Clients don't need to implement this
}

ttv::CoreAPI::CoreAPI() : mLocalLanguage("en") {
  mSettingRepository = std::make_shared<SettingRepository>();
}

TTV_ErrorCode ttv::CoreAPI::SetListener(std::shared_ptr<ICoreAPIListener> listener) {
  if (mState != State::Uninitialized) {
    return TTV_EC_SHUT_DOWN;
  }

  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::RegisterClient(std::shared_ptr<ICoreApiClient> client) {
  if (client == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  if (mState == State::Uninitialized || mState == State::ShuttingDown) {
    return TTV_EC_SHUT_DOWN;
  }

  auto iter = std::find(mApiClients.begin(), mApiClients.end(), client);
  if (iter != mApiClients.end()) {
    return TTV_EC_INVALID_ARG;
  }

  mApiClients.push_back(client);

  // Notify the new client of logged in users
  std::vector<std::shared_ptr<User>> users;
  mUserRepository->GetUsers(users);
  for (auto user : users) {
    client->CoreUserLoggedIn(user);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::UnregisterClient(std::shared_ptr<ICoreApiClient> client) {
  auto iter = std::find(mApiClients.begin(), mApiClients.end(), client);
  if (iter == mApiClients.end()) {
    return TTV_EC_INVALID_ARG;
  }

  mApiClients.erase(iter);

  // Allow the client to clean up data on users
  std::vector<std::shared_ptr<User>> users;
  mUserRepository->GetUsers(users);
  for (auto user : users) {
    client->CoreUserLoggedOut(user);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::SetGlobalSetting(const std::string& key, const std::string& value) {
  mSettingRepository->SetSetting(key, value);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::RemoveGlobalSetting(const std::string& key) {
  bool found = mSettingRepository->RemoveSetting(key);
  return found ? TTV_EC_SUCCESS : TTV_EC_INVALID_ARG;
}

TTV_ErrorCode ttv::CoreAPI::GetGlobalSetting(const std::string& key, std::string& value) {
  bool found = mSettingRepository->GetSetting(key, value);
  return found ? TTV_EC_SUCCESS : TTV_EC_INVALID_ARG;
}

TTV_ErrorCode ttv::CoreAPI::GetSubscribedPubsubTopics(std::vector<std::string>& subscribedTopics) {
  std::vector<std::shared_ptr<User>> users;
  mUserRepository->GetUsers(users);
  if (auto anonymousUser = mUserRepository->GetAnonymousUser()) {
    users.push_back(anonymousUser);
  }

  for (const auto& user : users) {
    if (auto pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>()) {
      auto topics = pubsub->GetSubscribedTopics();
      subscribedTopics.insert(subscribedTopics.end(), topics.begin(), topics.end());
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::CrashAbort() {
  abort();
}

std::string ttv::CoreAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::CoreAPI::Initialize(const InitializeCallback& callback) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // We need the client id to be set before initializing
  if (gTwitchClientId[0] == '\0') {
    ec = TTV_EC_INVALID_CLIENTID;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = ModuleBase::Initialize(callback);
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = CreateMainEventScheduler(mMainEventScheduler);

    // TODO: https://jira.twitch.com/browse/SDK-380
    // Remove once all platforms have MainEventScheduler implementations.
    if (TTV_FAILED(ec)) {
      mPollingEventScheduler = std::make_shared<PollingEventScheduler>();
      mMainEventScheduler = mPollingEventScheduler;
      ec = TTV_EC_SUCCESS;
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    mTaskRunner = std::make_shared<TaskRunner>("CoreAPI");
    mUserListener = std::make_shared<UserListener>(this);
    mPubSubListener = std::make_shared<PubSubListener>(this);

    mChannelRepository = std::make_shared<ChannelRepository>();
    mChannelRepository->SetTaskRunner(mTaskRunner);
    mChannelRepository->Initialize();

    mUserRepository = std::make_shared<UserRepository>();
    mUserRepository->SetTaskRunner(mTaskRunner);
    mUserRepository->Initialize();

    mTrackingContext = std::make_shared<TrackingContext>();

    mInternalData = std::make_shared<CoreAPIInternalData>();
    CreateMutex(mInternalData->mutex, "CoreAPIInternalData");

    InitializeAnonymousUser();

    mState = State::Initializing;
    NotifyStateChange();

    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

void ttv::CoreAPI::InitializeAnonymousUser() {
  auto user = mUserRepository->GetAnonymousUser();

  // Manually append a pubsub component to the anonymous user
  auto pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (pubsub == nullptr) {
    pubsub = std::make_shared<PubSubClient>(user, mSettingRepository);
    pubsub->SetTaskRunner(mTaskRunner);
    pubsub->SetMainEventScheduler(mMainEventScheduler);
    pubsub->AddListener(mPubSubListener);
    pubsub->Initialize();

    user->GetComponentContainer()->SetComponent(PubSubClient::GetComponentName(), pubsub);

    pubsub->SetConnectionPreference(PubSubClient::ConnectionPreference::OnDemand);
  }
}

TTV_ErrorCode ttv::CoreAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (!mApiClients.empty()) {
    ec = TTV_EC_STILL_IN_USE;
  } else {
    ec = ModuleBase::Shutdown(callback);
  }

  // Dispose all outstanding public components
  if (mInternalData != nullptr) {
    while (!mInternalData->channelStatusInstances.empty()) {
      mInternalData->channelStatusInstances[0]->Dispose();
    }

    while (!mInternalData->dashboardActivityStatusInstances.empty()) {
      mInternalData->dashboardActivityStatusInstances[0]->Dispose();
    }

    while (!mInternalData->genericSubscriberStatusInstances.empty()) {
      mInternalData->genericSubscriberStatusInstances[0]->Dispose();
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    RegisterShutdownCallback(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::CoreAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  if (mUserRepository != nullptr) {
    mUserRepository->Update();
  }

  if (mChannelRepository != nullptr) {
    mChannelRepository->Update();
  }

  // TODO: https://jira.twitch.com/browse/SDK-380
  // Remove once all platforms have MainEventScheduler implementations.
  if (mPollingEventScheduler != nullptr) {
    mPollingEventScheduler->Update();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::CoreAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (mUserRepository != nullptr) {
    if (mUserRepository->GetState() == UserRepository::State::Initialized) {
      mUserRepository->Shutdown();
    }

    if (mUserRepository->GetState() != UserRepository::State::Uninitialized) {
      return false;
    }
  }

  if (mChannelRepository != nullptr) {
    if (mChannelRepository->GetState() == ChannelRepository::State::Initialized) {
      mChannelRepository->Shutdown();
    }

    if (mChannelRepository->GetState() != ChannelRepository::State::Uninitialized) {
      return false;
    }
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  if (mMainEventScheduler != nullptr &&
      mMainEventScheduler->GetState() != IEventScheduler::EventSchedulerState::ShutDown) {
    if (mMainEventScheduler->GetState() == IEventScheduler::EventSchedulerState::Running) {
      mMainEventScheduler->Shutdown([this] { mMainEventScheduler.reset(); });
    }

    return false;
  }

  return true;
}

void ttv::CoreAPI::CompleteShutdown() {
  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  mPollingEventScheduler.reset();
  mUserRepository.reset();
  mChannelRepository.reset();
  mUserRepository.reset();
  mInternalData.reset();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::CoreAPI::GetClientId(std::string& clientId) {
  clientId = gTwitchClientId;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::SetLocalLanguage(const std::string& language) {
  TTV_RETURN_ON_EMPTY_STDSTRING(language, TTV_EC_INVALID_ARG);

  if (language != mLocalLanguage) {
    mLocalLanguage = language;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::GetLocalLanguage(std::string& language) {
  language = mLocalLanguage;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::FetchUserInfoById(UserId userId, FetchUserInfoCallback callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mUserRepository->FetchUserInfoById(userId, callback);
}

TTV_ErrorCode ttv::CoreAPI::FetchUserInfoByName(const std::string& userName, FetchUserInfoCallback callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_DIFFERENT(IsValidUserName(userName), true, TTV_EC_INVALID_LOGIN);

  return mUserRepository->FetchUserInfoByName(userName, callback);
}

TTV_ErrorCode ttv::CoreAPI::FetchChannelInfoById(ChannelId channelId, const FetchChannelInfoCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mChannelRepository->FetchChannelInfo(channelId, callback);
}

TTV_ErrorCode ttv::CoreAPI::FetchChannelInfoByName(
  const std::string& channelName, const FetchChannelInfoCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_DIFFERENT(IsValidChannelName(channelName), true, TTV_EC_INVALID_ARG);

  UserInfo userInfo;
  if (TTV_SUCCEEDED(mUserRepository->GetUserInfoByName(channelName, userInfo))) {
    return FetchChannelInfoById(userInfo.userId, callback);
  } else {
    return mUserRepository->FetchUserInfoByName(
      channelName, [this, callback](const ErrorDetails& errorDetails, const UserInfo& fetchedUserInfo) {
        if (TTV_SUCCEEDED(errorDetails.ec)) {
          FetchChannelInfoById(fetchedUserInfo.userId, callback);
        } else {
          callback(errorDetails.ec, {});
        }
      });
  }
}

TTV_ErrorCode ttv::CoreAPI::FetchStreamInfoById(ChannelId channelId, const FetchStreamInfoCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);

  GetStreamTask::Callback fetchCallback = [callback](GetStreamTask* /*source*/, TTV_ErrorCode ec,
                                            const std::shared_ptr<GetStreamTask::Result>& result) {
    if (callback != nullptr) {
      callback(ec, *result->streamInfo);
    }
  };

  // TODO: We might want to do this on behalf of a user to pass an OAuth token
  auto task = std::make_shared<GetStreamTask>(channelId, "", fetchCallback);

  if (mTaskRunner->AddTask(task)) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUTTING_DOWN;
  }
}

TTV_ErrorCode ttv::CoreAPI::FetchStreamInfoByName(
  const std::string& channelName, const FetchStreamInfoCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_DIFFERENT(IsValidChannelName(channelName), true, TTV_EC_INVALID_ARG);

  GetStreamTask::Callback fetchCallback = [callback](GetStreamTask* /*source*/, TTV_ErrorCode ec,
                                            const std::shared_ptr<GetStreamTask::Result>& result) {
    if (callback != nullptr) {
      callback(ec, *result->streamInfo);
    }
  };

  // TODO: We might want to do this on behalf of a user to pass an OAuth token
  auto task = std::make_shared<GetStreamTask>(channelName, "", fetchCallback);

  if (mTaskRunner->AddTask(task)) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUTTING_DOWN;
  }
}

TTV_ErrorCode ttv::CoreAPI::CreateChannelStatus(UserId userId, ChannelId channelId,
  const std::shared_ptr<IChannelListener>& listener, std::shared_ptr<IChannelStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  auto channelStatus = std::make_shared<ChannelStatus>(user, channelId);

  channelStatus->SetListener(listener);
  channelStatus->SetTaskRunner(mTaskRunner);
  TTV_ErrorCode ec = channelStatus->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    channelStatus->SetDisposer([channelStatus, data = mInternalData]() { DisposeChannelStatus(channelStatus, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->channelStatusInstances.push_back(channelStatus);
    }

    user->GetComponentContainer()->AddComponent(channelStatus);

    result.reset(channelStatus.get(), [channelStatus](IChannelStatus* /*p*/) { channelStatus->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::CoreAPI::DisposeChannelStatus(
  const std::shared_ptr<IChannelStatus>& channelStatus, const std::shared_ptr<CoreAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(channelStatus, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter =
    std::find(internalData->channelStatusInstances.begin(), internalData->channelStatusInstances.end(), channelStatus);
  if (iter != internalData->channelStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->channelStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::CreateDashboardActivityStatus(UserId userId, ChannelId channelId,
  const std::shared_ptr<IDashboardActivityListener>& listener, std::shared_ptr<IDashboardActivityStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  user = mUserRepository->GetUser(userId);

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto status = std::make_shared<DashboardActivityStatus>(user, channelId);

  status->SetListener(listener);
  status->SetTaskRunner(mTaskRunner);
  TTV_ErrorCode ec = status->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    status->SetDisposer([status, data = mInternalData]() { DisposeDashboardActivityStatus(status, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->dashboardActivityStatusInstances.push_back(status);
    }

    user->GetComponentContainer()->AddComponent(status);

    result.reset(status.get(), [status](IDashboardActivityStatus* /*p*/) { status->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::CoreAPI::CreateGenericSubscriberStatus(UserId userId, const std::string& subscriberTopic,
  const std::shared_ptr<IGenericSubscriberListener>& listener, std::shared_ptr<IGenericSubscriberStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  auto status = std::make_shared<GenericSubscriberStatus>(user, subscriberTopic);

  status->SetListener(listener);
  status->SetTaskRunner(mTaskRunner);
  TTV_ErrorCode ec = status->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    status->SetDisposer([status, data = mInternalData]() { DisposeGenericSubscriberStatus(status, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->genericSubscriberStatusInstances.push_back(status);
    }

    user->GetComponentContainer()->AddComponent(status);

    result.reset(status.get(), [status](IGenericSubscriberStatus* /*p*/) { status->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::CoreAPI::DisposeDashboardActivityStatus(
  const std::shared_ptr<IDashboardActivityStatus>& status, const std::shared_ptr<CoreAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(status, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->dashboardActivityStatusInstances.begin(),
    internalData->dashboardActivityStatusInstances.end(), status);
  if (iter != internalData->dashboardActivityStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->dashboardActivityStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::DisposeGenericSubscriberStatus(
  const std::shared_ptr<IGenericSubscriberStatus>& status, const std::shared_ptr<CoreAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(status, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->genericSubscriberStatusInstances.begin(),
    internalData->genericSubscriberStatusInstances.end(), status);
  if (iter != internalData->genericSubscriberStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->genericSubscriberStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::GetRequiredOAuthScopes(
  std::vector<std::string>& modules, std::vector<std::string>& allScopes) {
  // Insert the ones needed by CoreAPI
  allScopes.insert(
    allScopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));

  // Build the scope list
  for (auto client : mApiClients) {
    modules.push_back(client->GetClientName());

    std::vector<std::string> scopes;
    client->GetRequiredOAuthScopes(scopes);

    for (size_t i = 0; i < scopes.size(); ++i) {
      if (std::find(allScopes.begin(), allScopes.end(), scopes[i]) == allScopes.end()) {
        allScopes.push_back(scopes[i]);
      }
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::CoreAPI::LogIn(const std::string& oauthToken, LogInCallback callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_DIFFERENT(IsValidOAuthToken(oauthToken), true, TTV_EC_INVALID_LOGIN);

  return mUserRepository->FetchUserInfoByAuthToken(std::make_shared<OAuthToken>(oauthToken),
    [this, oauthToken, callback](const ErrorDetails& errorDetails, const UserInfo& info) {
      if (TTV_SUCCEEDED(errorDetails.ec)) {
        auto userId = info.userId;
        auto user = mUserRepository->GetUser(userId);
        if (user == nullptr) {
          user = mUserRepository->RegisterUser(userId);
          user->SetUserRepository(mUserRepository);
          user->AddListener(mUserListener);

          // Create the PubSubClient instance
          auto pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>();
          if (pubsub == nullptr) {
            pubsub = std::make_shared<PubSubClient>(user, mSettingRepository);
            pubsub->SetTaskRunner(mTaskRunner);
            pubsub->SetMainEventScheduler(mMainEventScheduler);
            pubsub->AddListener(mPubSubListener);
            pubsub->Initialize();

            user->GetComponentContainer()->SetComponent(PubSubClient::GetComponentName(), pubsub);

            // Configure pubsub to only connect when there are modules using it
            pubsub->SetConnectionPreference(PubSubClient::ConnectionPreference::OnDemand);
            // pubsub->Connect();
          }
        }

        user->SetOAuthToken(std::make_shared<OAuthToken>(oauthToken));
        user->SetUserInfo(info);
        user->LogIn();
      }

      if (callback != nullptr) {
        callback(errorDetails, info);
      }

      Invoke<ICoreAPIListener>([&oauthToken, &info, &errorDetails](const std::shared_ptr<ICoreAPIListener>& listener) {
        listener->CoreUserLoginComplete(oauthToken, info.userId, errorDetails);
      });
    });
}

TTV_ErrorCode ttv::CoreAPI::LogOut(UserId userId, LogOutCallback callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_LOGIN);

  // NOTE: Removing a user from the repository can cause collection of associated data to occur if a reference isn't
  // retained. Any components still using the user should ensure that they keep a reference to the user until done with
  // it.

  // This will in turn call OnUserLogOutComplete and notify API clients
  TTV_ErrorCode ec = mUserRepository->UnRegisterUser(userId);

  if (TTV_SUCCEEDED(ec)) {
    LambdaTask::CompleteCallback completeCallback = [this, userId, callback](
                                                      LambdaTask* /*source*/, TTV_ErrorCode callbackEc) {
      if (callback != nullptr) {
        callback(callbackEc);
      }

      Invoke<ICoreAPIListener>([userId, callbackEc](const std::shared_ptr<ICoreAPIListener>& listener) {
        listener->CoreUserLogoutComplete(userId, callbackEc);
      });
    };

    std::shared_ptr<Task> task = std::make_shared<LambdaTask>(nullptr, completeCallback);
    if (mTaskRunner->AddTask(task)) {
      return TTV_EC_SUCCESS;
    } else {
      return TTV_EC_SHUTTING_DOWN;
    }
  }
  return ec;
}

TTV_ErrorCode ttv::CoreAPI::ConnectPubSub(UserId userId) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  std::shared_ptr<ttv::User> user;
  if (userId == 0) {
    user = mUserRepository->GetAnonymousUser();
  } else {
    user = mUserRepository->GetUser(userId);
  }

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (pubsub == nullptr) {
    return TTV_EC_SHUT_DOWN;
  }

  return pubsub->Connect();
}

TTV_ErrorCode ttv::CoreAPI::DisconnectPubSub(UserId userId) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  std::shared_ptr<ttv::User> user;
  if (userId == 0) {
    user = mUserRepository->GetAnonymousUser();
  } else {
    user = mUserRepository->GetUser(userId);
  }

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (pubsub == nullptr) {
    return TTV_EC_SHUT_DOWN;
  }

  return pubsub->Disconnect();
}

void ttv::CoreAPI::OnPubSubStateChanged(PubSubClient* source, PubSubState state, TTV_ErrorCode ec) {
  auto user = source->GetUser();
  if (user == nullptr) {
    return;
  }

  // Notify external listeners
  Invoke<ICoreAPIListener>([user, state, ec](std::shared_ptr<ICoreAPIListener> listener) {
    listener->CorePubSubStateChanged(user->GetUserId(), state, ec);
  });
}

void ttv::CoreAPI::OnUserLogInComplete(User* source, TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec)) {
    auto user = source->shared_from_this();
    for (auto client : mApiClients) {
      client->CoreUserLoggedIn(user);
    }
  }
}

void ttv::CoreAPI::OnUserLogOutComplete(User* source, TTV_ErrorCode /*ec*/) {
  auto user = source->shared_from_this();
  for (auto client : mApiClients) {
    client->CoreUserLoggedOut(user);
  }
}

void ttv::CoreAPI::OnUserInfoFetchComplete(User* /*source*/, TTV_ErrorCode /*ec*/) {
  // NOOP
}

void ttv::CoreAPI::OnUserAuthenticationIssue(
  User* source, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec) {
  // Notify external listeners
  auto oauth = oauthToken->GetToken();
  ErrorDetails errorDetails(ec);
  Invoke<ICoreAPIListener>([source, &oauth, &errorDetails](std::shared_ptr<ICoreAPIListener> listener) {
    listener->CoreUserAuthenticationIssue(source->GetUserId(), oauth, errorDetails);
  });
}

ttv::CoreAPI::PubSubListener::PubSubListener(CoreAPI* api) : mOwner(api) {}

void ttv::CoreAPI::PubSubListener::OnStateChanged(PubSubClient* source, PubSubState state, TTV_ErrorCode ec) {
  mOwner->OnPubSubStateChanged(source, state, ec);
}

ttv::CoreAPI::UserListener::UserListener(CoreAPI* api) : mOwner(api) {}

void ttv::CoreAPI::UserListener::OnUserLogInComplete(User* source, TTV_ErrorCode ec) {
  mOwner->OnUserLogInComplete(source, ec);
}

void ttv::CoreAPI::UserListener::OnUserLogOutComplete(User* source, TTV_ErrorCode ec) {
  mOwner->OnUserLogOutComplete(source, ec);
}

void ttv::CoreAPI::UserListener::OnUserInfoFetchComplete(User* source, TTV_ErrorCode ec) {
  mOwner->OnUserInfoFetchComplete(source, ec);
}

void ttv::CoreAPI::UserListener::OnUserAuthenticationIssue(
  User* source, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec) {
  mOwner->OnUserAuthenticationIssue(source, oauthToken, ec);
}
