/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/dashboardactivitystatus.h"

#include "twitchsdk/core/json/jsonobjectdescriptions.h"

namespace {
const char* kLoggerName = "DashboardActivityStatus";
const char* kTopicPrefix = "dashboard-activity-feed.";

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityBitsUsage& data) {
  if (!(ttv::json::ToObject(jMessage, data.header) && ttv::json::ToObject(jMessage, data))) {
    return false;
  }

  // user can be anonymous
  return jMessage["bits_user"].isNull() || ttv::json::ToObject(jMessage["bits_user"], data.user);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityFollow& data) {
  return ttv::json::ToObject(jMessage, data.header) && ttv::json::ToObject(jMessage["follower"], data.follower);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityHost& data) {
  return ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header) &&
         ttv::json::ToObject(jMessage["host"], data.host);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityRaiding& data) {
  return ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header) &&
         ttv::json::ToObject(jMessage["raider"], data.raider);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivitySubscription& data) {
  return ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header) &&
         ttv::json::ToObject(jMessage["subscriber"], data.subscriber);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityResubscriptionSharing& data) {
  if (!(ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header) &&
        ttv::json::ToObject(jMessage["subscriber"], data.subscriber))) {
    return false;
  }

  const auto& jFragments = jMessage["subscription_custom_message_fragments"];

  if (!jFragments.isArray() || jFragments.empty()) {
    return true;
  }

  for (const auto& jFragment : jFragments) {
    ttv::DashboardActivityFragment element;

    if (!ttv::json::ToObject(jFragment, element)) {
      return false;
    }

    data.customMessageFragments.push_back(element);
  }

  return true;
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivitySubscriptionGiftingCommunity& data) {
  if (!(ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header))) {
    return false;
  }

  // user can be anonymous
  return jMessage["subscription_gifter"].isNull() || ttv::json::ToObject(jMessage["subscription_gifter"], data.gifter);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivitySubscriptionGiftingIndividual& data) {
  if (!(ttv::json::ToObject(jMessage, data) && ttv::json::ToObject(jMessage, data.header) &&
        ttv::json::ToObject(jMessage["subscription_gift_recipient"], data.recipient))) {
    return false;
  }

  // user can be anonymous
  return jMessage["subscription_gifter"].isNull() || ttv::json::ToObject(jMessage["subscription_gifter"], data.gifter);
}

bool Parse(const ttv::json::Value& jMessage, ttv::DashboardActivityHeader& data) {
  return ttv::json::ToObject(jMessage, data);
}
}  // namespace

ttv::DashboardActivityStatus::DashboardActivityStatus(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(channelId)) {
  AddTopic(mPubSubTopic);
}

std::string ttv::DashboardActivityStatus::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::DashboardActivityStatus::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::DashboardActivityStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "DashboardActivityStatusListener SubscribeStateChanged: %s",
    PubSubClient::SubscribeState::ToString(state).c_str());
}

void ttv::DashboardActivityStatus::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jRoot) {
  if (topic != mPubSubTopic) {
    return;
  }

  if (mListener == nullptr) {
    Log(MessageLevel::Error, "Listener not initialized");
    return;
  }

  if (jRoot.isNull() || !jRoot.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  const auto& jType = jRoot["type"];
  if (jType.isNull() || !jType.isString()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  const auto type = jType.asString();
  if (type == "bits_usage") {
    HandleBitsUseage(jRoot);
  } else if (type == "follow") {
    HandleFollow(jRoot);
  } else if (type == "auto_host_start") {
    HandleAutoHostStart(jRoot);
  } else if (type == "host_start") {
    HandleHostStart(jRoot);
  } else if (type == "raiding") {
    HandleRaiding(jRoot);
  } else if (type == "prime_subscription") {
    HandlePrimeSubscription(jRoot);
  } else if (type == "subscription") {
    HandleSubscription(jRoot);
  } else if (type == "prime_resubscription_sharing") {
    HandlePrimeResubscriptionSharing(jRoot);
  } else if (type == "resubscription_sharing") {
    HandleResubscriptionSharing(jRoot);
  } else if (type == "subscription_gifting_community") {
    HandleSubscriptionGiftingCommunity(jRoot);
  } else if (type == "subscription_gifting_individual") {
    HandleSubscriptionGiftingIndividual(jRoot);
  } else if (type == "stream_up") {
    HandleStreamUp(jRoot);
  } else {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
  }
}

void ttv::DashboardActivityStatus::HandleBitsUseage(const ttv::json::Value& jMessage) {
  DashboardActivityBitsUsage data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleBitsUseage");
    return;
  }

  mListener->EventBitsUseage(data);
}

void ttv::DashboardActivityStatus::HandleFollow(const ttv::json::Value& jMessage) {
  DashboardActivityFollow data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleFollow");
    return;
  }

  mListener->EventFollower(data);
}

void ttv::DashboardActivityStatus::HandleAutoHostStart(const ttv::json::Value& jMessage) {
  DashboardActivityHost data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleAutoHostStart");
    return;
  }

  mListener->EventAutoHostStart(data);
}

void ttv::DashboardActivityStatus::HandleHostStart(const ttv::json::Value& jMessage) {
  DashboardActivityHost data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleHostStart");
    return;
  }

  mListener->EventHostStart(data);
}

void ttv::DashboardActivityStatus::HandleRaiding(const ttv::json::Value& jMessage) {
  DashboardActivityRaiding data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleRaiding");
    return;
  }

  mListener->EventRaiding(data);
}

void ttv::DashboardActivityStatus::HandlePrimeSubscription(const ttv::json::Value& jMessage) {
  DashboardActivitySubscription data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandlePrimeSubscription");
    return;
  }

  mListener->EventPrimeSubscription(data);
}

void ttv::DashboardActivityStatus::HandleSubscription(const ttv::json::Value& jMessage) {
  DashboardActivitySubscription data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleSubscription");
    return;
  }

  mListener->EventSubscription(data);
}

void ttv::DashboardActivityStatus::HandlePrimeResubscriptionSharing(const ttv::json::Value& jMessage) {
  DashboardActivityResubscriptionSharing data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandlePrimeResubscriptionSharing");
    return;
  }

  mListener->EventPrimeResubscriptionSharing(data);
}

void ttv::DashboardActivityStatus::HandleResubscriptionSharing(const ttv::json::Value& jMessage) {
  DashboardActivityResubscriptionSharing data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleResubscriptionSharing");
    return;
  }

  mListener->EventResubscriptionSharing(data);
}

void ttv::DashboardActivityStatus::HandleSubscriptionGiftingCommunity(const ttv::json::Value& jMessage) {
  DashboardActivitySubscriptionGiftingCommunity data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleSubscriptionGiftingCommunity");
    return;
  }

  mListener->EventSubscriptionGiftingCommunity(data);
}

void ttv::DashboardActivityStatus::HandleSubscriptionGiftingIndividual(const ttv::json::Value& jMessage) {
  DashboardActivitySubscriptionGiftingIndividual data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleSubscriptionGiftingIndividual");
    return;
  }

  mListener->EventSubscriptionGiftingIndividual(data);
}

void ttv::DashboardActivityStatus::HandleStreamUp(const ttv::json::Value& jMessage) {
  DashboardActivityHeader data;

  if (!Parse(jMessage, data)) {
    Log(MessageLevel::Error, "Could not parse json: DashboardActivityStatus::HandleStreamUp");
    return;
  }

  mListener->EventStreamUp(data);
}
