/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/types/errortypes.h"

namespace {
std::vector<ttv::ErrorToStringFunction> gErrorToStringFunctions;
std::vector<ttv::ErrorCodeValueFunction> gErrorCodeValueFunctions;

// TODO: Consider using a mutex in case a module is registering/unregistering during lookup
}  // namespace

void ttv::RegisterErrorToStringFunction(ttv::ErrorToStringFunction func) {
  auto iter = std::find(gErrorToStringFunctions.begin(), gErrorToStringFunctions.end(), func);
  if (iter == gErrorToStringFunctions.end()) {
    gErrorToStringFunctions.push_back(func);
  }
}

void ttv::UnregisterErrorToStringFunction(ttv::ErrorToStringFunction func) {
  auto iter = std::find(gErrorToStringFunctions.begin(), gErrorToStringFunctions.end(), func);
  if (iter != gErrorToStringFunctions.end()) {
    gErrorToStringFunctions.erase(iter);
  }
}

const char* ttv::ErrorToString(TTV_ErrorCode ec) {
  const char* result = ttv::CoreErrorToString(ec);

  if (result == nullptr) {
    // Create a copy to make it slightly more thread safe.
    auto copy = gErrorToStringFunctions;

    for (auto& func : copy) {
      result = func(ec);

      if (result != nullptr) {
        return result;
      }
    }

    result = "unknown error";
  }

  return result;
}

const char* ttv::CoreErrorToString(TTV_ErrorCode err) {
  const char* result = nullptr;

  switch (err) {
#define USE_TTV_ERROR_ID(ec) \
  case ec:                   \
    result = #ec;            \
    break;
#define USE_TTV_SUCCESS_ID(ec) \
  case ec:                     \
    result = #ec;              \
    break;
#define BEGIN_ERROR_IDS(ec) \
  case ec:                  \
    result = #ec;           \
    break;
#define END_ERROR_IDS(ec) \
  case ec:                \
    result = #ec;         \
    break;
    TTV_CORE_ERROR_IDS
#undef BEGIN_ERROR_IDS
#undef END_ERROR_IDS
#undef USE_TTV_SUCCESS_ID
#undef USE_TTV_ERROR_ID
    default:
      break;
  }

  return result;
}

ttv::EnumValue::EnumValue(const char* nameArg, uint32_t valueArg) : name(nameArg), value(valueArg) {}

ttv::ErrorDetails::ErrorDetails() : details(json::objectValue), ec(TTV_EC_SUCCESS) {}

ttv::ErrorDetails::ErrorDetails(TTV_ErrorCode errorCode) : details(json::objectValue), ec(errorCode) {}

ttv::ErrorDetails::ErrorDetails(TTV_ErrorCode errorCode, const std::string& message)
    : details(json::objectValue), ec(errorCode) {
  if (!message.empty()) {
    details["Message"] = message;
  }
}

ttv::json::Value& ttv::ErrorDetails::operator[](const char* const key) {
  TTV_ASSERT(!details.isMember(key));
  return details[key];
}

ttv::ErrorDetails& ttv::ErrorDetails::operator=(TTV_ErrorCode other) {
  ec = other;
  return *this;
}

void ttv::GetCoreErrorCodeValues(std::vector<EnumValue>& result) {
#define USE_TTV_ERROR_ID(ec) result.emplace_back(#ec, static_cast<uint32_t>(ec));
#define USE_TTV_SUCCESS_ID(ec) result.emplace_back(#ec, static_cast<uint32_t>(ec));
#define BEGIN_ERROR_IDS(ec) result.emplace_back(#ec, static_cast<uint32_t>(ec));
#define END_ERROR_IDS(ec) result.emplace_back(#ec, static_cast<uint32_t>(ec));
  TTV_CORE_ERROR_IDS
#undef BEGIN_ERROR_IDS
#undef END_ERROR_IDS
#undef USE_TTV_SUCCESS_ID
#undef USE_TTV_ERROR_ID
}

void ttv::RegisterErrorCodeValueFunction(ErrorCodeValueFunction func) {
  auto iter = std::find(gErrorCodeValueFunctions.begin(), gErrorCodeValueFunctions.end(), func);
  if (iter == gErrorCodeValueFunctions.end()) {
    gErrorCodeValueFunctions.push_back(func);
  }
}

void ttv::UnregisterErrorCodeValueFunction(ErrorCodeValueFunction func) {
  auto iter = std::find(gErrorCodeValueFunctions.begin(), gErrorCodeValueFunctions.end(), func);
  if (iter != gErrorCodeValueFunctions.end()) {
    gErrorCodeValueFunctions.erase(iter);
  }
}

void ttv::GetAllErrorCodes(std::vector<EnumValue>& result) {
  ttv::GetCoreErrorCodeValues(result);

  // Create a copy to make it slightly more thread safe.
  auto copy = gErrorCodeValueFunctions;

  for (const auto& func : copy) {
    func(result);
  }
}
