/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/eventqueue.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/result.h"
#include "twitchsdk/core/systemclock.h"

namespace {
const char* kLoggerName = "EventQueue";
}

ttv::EventQueue::EventQueue(const std::string& name) : mName("EventQueue-" + name), mCurrentTaskId(1) {
  TTV_ErrorCode ec = ttv::CreateConditionMutex(mMutex, mName);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mMutex != nullptr);
}

ttv::EventQueue::~EventQueue() {
  TTV_ASSERT(mQueue.empty());
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s destroyed", mName.c_str());
}

ttv::TaskId ttv::EventQueue::InsertTask(TaskParams&& taskParams) {
  TTV_ASSERT(taskParams.taskFunc != nullptr);

  TaskId taskId;
  std::string taskName = taskParams.taskName;

  Task task;
  task.taskFunc = std::move(taskParams.taskFunc);
  task.taskName = std::move(taskParams.taskName);
  task.invocationTimestamp = GetSystemTimeMilliseconds() + taskParams.delayMilliseconds;

  {
    AutoMutex lock(mMutex.get());
    taskId = mCurrentTaskId;
    mCurrentTaskId++;
    task.taskId = taskId;

    auto it = std::upper_bound(mQueue.begin(), mQueue.end(), task);
    mQueue.insert(it, std::move(task));
  }
  mMutex->Signal();

  ttv::trace::Message(
    kLoggerName, MessageLevel::Debug, "EventQueue %s: Task \"%s\" (%d) added", mName.c_str(), taskName.c_str(), taskId);

  return taskId;
}

bool ttv::EventQueue::RemoveTask(TaskId taskId) {
  bool taskRemoved = false;

  {
    AutoMutex lock(mMutex.get());

    auto queueIt =
      std::find_if(mQueue.begin(), mQueue.end(), [taskId](const Task& task) { return task.taskId == taskId; });

    if (queueIt != mQueue.end()) {
      if (queueIt->taskFunc != nullptr) {
        // Move cancelled tasks to be run in WaitForEvent() to ensure destructors are called in the right thread.
        mCancelledTasks.emplace_back(std::move(queueIt->taskFunc));
        queueIt->taskFunc = nullptr;
        taskRemoved = true;
      }
    }
  }

  if (taskRemoved) {
    // Notify that a task was removed
    mMutex->Signal();
    ttv::trace::Message(
      kLoggerName, MessageLevel::Debug, "EventQueue %s: Task (%d) successfully removed", mName.c_str(), taskId);
  } else {
    ttv::trace::Message(
      kLoggerName, MessageLevel::Debug, "EventQueue %s: Task (%d) unsuccessfully removed", mName.c_str(), taskId);
  }

  return taskRemoved;
}

void ttv::EventQueue::Clear() {
  {
    AutoMutex lock(mMutex.get());
    mQueue.clear();
  }

  mMutex->Signal();
}

void ttv::EventQueue::WaitForEvent() {
  bool ranTask = false;

  while (!ranTask) {
    // Clear out cancelled tasks and their associated lambdas - with any possible destructors.
    // We do this in WaitForEvent() and not in RemoveTask() to ensure destructors are called in the right thread.
    std::vector<TaskFunc> tasksToClear;
    {
      AutoMutex lock(mMutex.get());
      tasksToClear.swap(mCancelledTasks);
    }

    tasksToClear.clear();

    Task task;

    {
      AutoMutex lock(mMutex.get());

      uint64_t currentTime = GetSystemTimeMilliseconds();

      auto it = mQueue.begin();
      if (it != mQueue.end()) {
        if (it->invocationTimestamp <= currentTime) {
          // Remove the task from the EventQueue, and save the taskFunc to run outside the lock
          if (it->taskFunc != nullptr) {
            task = std::move(*it);
          }
          mQueue.erase(it);
        } else {
          // Wait until the first event is ready to fire
          uint64_t waitTime = it->invocationTimestamp - currentTime;
          ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Suspending for %dms to wait for tasks",
            mName.c_str(), waitTime);
          mMutex->WaitFor(waitTime);
        }
      } else {
        ttv::trace::Message(
          kLoggerName, MessageLevel::Debug, "EventQueue %s: Suspending indefinitely to wait for tasks", mName.c_str());
        mMutex->Wait();
      }
    }

    if (task.taskFunc != nullptr) {
      ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Running Task \"%s\" (%d)", mName.c_str(),
        task.taskName.c_str(), task.taskId);
      task.taskFunc();
      task.taskFunc = nullptr;
      ranTask = true;
    }
  }
}

bool ttv::EventQueue::WaitForEventWithTimeout(uint64_t timeout) {
  uint64_t currentTime = GetSystemTimeMilliseconds();
  if (currentTime >= std::numeric_limits<uint64_t>::max() - timeout) {
    // CurrentTime + timeout will overflow or hit the max, so we will wait without a timeout
    WaitForEvent();
    return true;
  }

  uint64_t endTime = currentTime + timeout;

  bool ranTask = false;
  bool forceTry = true;

  // Loop while there's still time remaining before the timeout
  // Also always try if it's the first time OR the last try we hit a cancelled task
  while (currentTime <= endTime || forceTry) {
    forceTry = false;

    // Clear out cancelled tasks and their associated lambdas - with any possible destructors.
    // We do this in WaitForEvent() and not in RemoveTask() to ensure destructors are called in the right thread.
    std::vector<TaskFunc> tasksToClear;
    {
      AutoMutex lock(mMutex.get());
      tasksToClear.swap(mCancelledTasks);
    }

    tasksToClear.clear();

    Task task;

    {
      AutoMutex lock(mMutex.get());

      currentTime = GetSystemTimeMilliseconds();

      auto it = mQueue.begin();
      if (it != mQueue.end()) {
        if (it->invocationTimestamp <= currentTime) {
          // Remove the task from the EventQueue, and save the taskFunc to run outside the lock
          if (it->taskFunc != nullptr) {
            task = std::move(*it);
          } else {
            // Force another loop iteration since we're just cleaning up a cancelled task in this iteration
            forceTry = true;
          }
          mQueue.erase(it);
        } else {
          // Wait difference from now until endTime OR now until first timestamp (whichever is shorter)
          if (endTime > currentTime) {
            uint64_t waitTime = endTime - currentTime;
            waitTime = std::min(waitTime, it->invocationTimestamp - currentTime);
            ttv::trace::Message(kLoggerName, MessageLevel::Debug,
              "EventQueue %s: Suspending for %dms to wait for tasks", mName.c_str(), waitTime);
            mMutex->WaitFor(waitTime);
          } else {
            // ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Timed out while wait for tasks",
            // mName.c_str());
            // The timeout has been reached
            break;
          }
        }
      } else {
        // Queue is empty, so wait the difference from now until endTime
        if (endTime > currentTime) {
          uint64_t waitTime = endTime - currentTime;
          ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Suspending for %dms to wait for tasks",
            mName.c_str(), waitTime);
          mMutex->WaitFor(waitTime);
        } else {
          // ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Timed out while wait for tasks",
          // mName.c_str());
          // The timeout has been reached
          break;
        }
      }
    }

    if (task.taskFunc != nullptr) {
      ttv::trace::Message(kLoggerName, MessageLevel::Debug, "EventQueue %s: Running Task \"%s\" (%d)", mName.c_str(),
        task.taskName.c_str(), task.taskId);
      task.taskFunc();
      task.taskFunc = nullptr;
      ranTask = true;
      break;
    }

    currentTime = GetSystemTimeMilliseconds();
  }

  return ranTask;
}
