/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/eventtracker.h"

namespace {
std::shared_ptr<ttv::IEventTracker> gEventTracker;

/**
 * Creates a human readable string from a tracking event, to log to the console.
 */
std::string StringForEvent(const std::string& eventName, const std::map<std::string, ttv::TrackingValue>& properties) {
  std::string eventString = "   Name: ";
  eventString.append(eventName);
  eventString.append("\n    Properties:\n");
  eventString.append("    {\n");

  bool isFirst = true;
  for (const auto& pair : properties) {
    if (isFirst) {
      isFirst = false;
    } else {
      eventString.append(",\n");
    }
    eventString.append("        \"");
    eventString.append(pair.first);
    eventString.append("\" : ");

    switch (pair.second.GetType()) {
      case ttv::TrackingValue::Type::Null:
        eventString.append("null");
        break;
      case ttv::TrackingValue::Type::Boolean:
        eventString.append(pair.second.GetBooleanValue() ? "true" : "false");
        break;
      case ttv::TrackingValue::Type::Integer:
        eventString.append(std::to_string(pair.second.GetIntegerValue()));
        break;
      case ttv::TrackingValue::Type::Double:
        eventString.append(std::to_string(pair.second.GetDoubleValue()));
        break;
      case ttv::TrackingValue::Type::String:
        eventString.append("\"");
        eventString.append(pair.second.GetStringValue());
        eventString.append("\"");
        break;
    }
  }

  eventString.append("\n    }\n");

  return eventString;
};
}  // namespace

void ttv::SetEventTracker(const std::shared_ptr<IEventTracker>& eventTracker) {
  gEventTracker = eventTracker;
}

std::shared_ptr<ttv::IEventTracker> ttv::GetEventTracker() {
  return gEventTracker;
}

TTV_ErrorCode ttv::TrackEvent(const std::string& eventName, const std::map<std::string, TrackingValue>& properties) {
  ttv::trace::Message(
    "EventTracker", MessageLevel::Debug, "Sending Tracking Event:\n%s", StringForEvent(eventName, properties).c_str());

  if (gEventTracker == nullptr) {
    return TTV_EC_NOT_AVAILABLE;
  }

  return gEventTracker->TrackEvent(eventName, properties);
}
