/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/httprequest.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/random.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/thread.h"

#include <cmath>

namespace ttv {
TTV_ErrorCode GetClientId(std::string& clientId);
}

namespace {
using namespace ttv;

const char* kClientIdHeaderName = "Client-ID";

// The global http request provider
std::shared_ptr<HttpRequest> gHttpRequest;

/**
 * Adds the Client-Id header if this represents a Twitch API endpoint and the header is missing.
 */
void AppendTwitchClientIdHeader(const std::string& url, std::vector<HttpParam>& headers) {
  if (IsTwitchEndpoint(url)) {
    std::string clientId;
    GetClientId(clientId);

    if (clientId != "" && !ContainsHttpParameter(headers, kClientIdHeaderName)) {
      headers.push_back(HttpParam(kClientIdHeaderName, clientId));
    }
  }
}
}  // namespace

namespace {
const uint kHttpStatusCategory_5XX_ServerError = 5;

const uint kHttpStatusCode_501_NotImplemented = 501;
const uint kHttpStatusCode_505_HttpVersionNotSupported = 505;

const uint kAbortAfterNumServerErrors = 2;

uint CalculateRetryDelay(uint attempt) {
  // Don't delay on first try
  if (attempt == 0) {
    return 0;
  }

  // Wait longer before the next try
  uint delay = static_cast<uint>(std::pow(2.0, static_cast<double>(attempt - 1))) * 1000;

  // Add a bit of randomness so all clients don't retry at the same time
  std::uniform_int_distribution<uint> distribution(0, delay - 1);
  delay += distribution(ttv::random::GetGenerator()) / 2;

  return delay;
}

bool IsStatusCategory(uint category, uint statusCode) {
  return (statusCode / 100) == category;
}

struct HttpRetryRequestData {
  ttv::HttpRequestHeadersCallback headersCallback;
  ttv::HttpRequestCallback responseCallback;
  void* clientUserdata;
  uint numServerErrors;
  bool allowRetry;
  bool done;
};

bool HttpRetriesHeadersCallback(uint statusCode, const std::map<std::string, std::string>& headers, void* userData) {
  HttpRetryRequestData* requestData = reinterpret_cast<HttpRetryRequestData*>(userData);

  bool is5XX = IsStatusCategory(kHttpStatusCategory_5XX_ServerError, statusCode);

  // Don't retry
  if (!requestData->allowRetry || !is5XX || statusCode == kHttpStatusCode_501_NotImplemented ||
      statusCode == kHttpStatusCode_505_HttpVersionNotSupported ||
      (requestData->numServerErrors + 1 >= kAbortAfterNumServerErrors))  // Consecutive 5XX errors
  {
    requestData->done = true;
  }
  // Attempt a retry
  else {
    // Track consecutive 5XX errors
    if (is5XX) {
      requestData->numServerErrors++;
    }

    requestData->done = false;
  }

  // Notify the client
  bool responseDesired = requestData->done;
  if (requestData->done) {
    if (requestData->headersCallback) {
      responseDesired = requestData->headersCallback(statusCode, headers, requestData->clientUserdata);
    }
  }

  return responseDesired;
}

void HttpRetriesResponseCallback(uint statusCode, const std::vector<char>& result, void* userData) {
  HttpRetryRequestData* requestData = reinterpret_cast<HttpRetryRequestData*>(userData);

  if (requestData->responseCallback) {
    requestData->responseCallback(statusCode, result, requestData->clientUserdata);
  }
}
}  // namespace

ttv::HttpParam::HttpParam(const std::string& name, const std::string& value) : paramName(name), paramValue(value) {}

ttv::HttpParam::HttpParam(const std::string& name, int value) : paramName(name) {
  char buffer[64];
  snprintf(buffer, sizeof(buffer), "%d", value);
  paramValue = buffer;
}

void ttv::SetHttpRequest(const std::shared_ptr<ttv::HttpRequest>& http) {
  gHttpRequest = http;
}

std::shared_ptr<ttv::HttpRequest> ttv::GetHttpRequest() {
  return gHttpRequest;
}

TTV_ErrorCode ttv::HttpThreadInit() {
  if (gHttpRequest != nullptr) {
    return gHttpRequest->ThreadInit();
  } else {
    return TTV_EC_NOT_INITIALIZED;
  }
}

TTV_ErrorCode ttv::SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData) {
  if (gHttpRequest != nullptr) {
    std::vector<HttpParam> headers = requestHeaders;
    AppendTwitchClientIdHeader(url, headers);

    return gHttpRequest->SendHttpRequest(requestName, url, headers, requestBody, requestBodySize, httpReqType,
      timeOutInSecs, numRetries, headersCallback, responseCallback, userData);
  } else {
    return TTV_EC_NOT_INITIALIZED;
  }
}

TTV_ErrorCode ttv::SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData) {
  if (gHttpRequest != nullptr) {
    std::vector<HttpParam> headers = requestHeaders;
    AppendTwitchClientIdHeader(url, headers);

    return gHttpRequest->SendHttpRequest(requestName, url, headers, requestBody, requestBodySize, httpReqType,
      timeOutInSecs, headersCallback, responseCallback, userData);
  } else {
    return TTV_EC_NOT_INITIALIZED;
  }
}

TTV_ErrorCode ttv::SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& urlParams, const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody,
  size_t requestBodySize, HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries,
  HttpRequestHeadersCallback headersCallback, HttpRequestCallback responseCallback, void* userData) {
  Uri uri(url);
  for (const auto& iter : urlParams) {
    uri.SetParam(iter.paramName, iter.paramName);
  }

  return SendHttpRequest(requestName, uri.GetUrl(), requestHeaders, requestBody, requestBodySize, httpReqType,
    timeOutInSecs, numRetries, headersCallback, responseCallback, userData);
}

TTV_ErrorCode ttv::HttpThreadShutdown() {
  if (gHttpRequest != nullptr) {
    return gHttpRequest->ThreadShutdown();
  } else {
    return TTV_EC_NOT_INITIALIZED;
  }
}

bool ttv::IsTwitchEndpoint(const std::string& url) {
  Uri uri(url);
  std::string host = ToLowerCase(uri.GetHostName());

  return host == "twitch.tv" || EndsWith(host, ".twitch.tv");
}

ttv::HttpRequest::HttpRequest() {}

ttv::HttpRequest::~HttpRequest() {}

TTV_ErrorCode ttv::HttpRequest::ThreadInit() {
  // NOP
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::HttpRequest::SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData) {
  HttpRetryRequestData data;
  data.numServerErrors = 0;
  data.headersCallback = headersCallback;
  data.responseCallback = responseCallback;
  data.clientUserdata = userData;
  data.done = false;

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  for (uint attempt = 0; attempt <= numRetries; ++attempt) {
    // Wait before trying
    uint delay = CalculateRetryDelay(attempt);
    if (delay > 0) {
      ttv::Sleep(delay);
    }

    // Whether or not the callback should defer notifying the client if a failure occurs
    data.allowRetry = attempt < numRetries;

    ec = SendHttpRequest(requestName, url, requestHeaders, requestBody, requestBodySize, httpReqType, timeOutInSecs,
      &HttpRetriesHeadersCallback, &HttpRetriesResponseCallback, reinterpret_cast<void*>(&data));
    if (TTV_FAILED(ec) || data.done) {
      break;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::HttpRequest::ThreadShutdown() {
  // NOP
  return TTV_EC_SUCCESS;
}
