/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/json/corejsonutil.h"

#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/user/user.h"

#include <sstream>

namespace {
template <typename ID_TYPE>
bool ParseUnsignedInt32(const ttv::json::Value& jId, ID_TYPE& result) {
  result = 0;

  if (jId.isNull()) {
    return false;
  } else if (jId.isString()) {
    return ttv::ParseNum(jId.asString(), result);
  } else if (jId.isNumeric()) {
    result = static_cast<uint32_t>(jId.asUInt());
    return true;
  } else {
    return false;
  }
}
}  // namespace

bool ttv::ParseString(const ttv::json::Value& root, const char* key, std::string& result) {
  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (!value.isString()) {
    return false;
  }

  result = value.asString();
  return true;
}

bool ttv::ParseString(
  const ttv::json::Value& root, const std::string& key, std::string& result, const char* defaultValue) {
  result = defaultValue;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  }

  if (!value.isString()) {
    return false;
  }

  result = value.asString();
  return true;
}

bool ttv::ParseBool(const ttv::json::Value& root, const char* key, bool& result) {
  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return false;
  }

  if (value.isString()) {
    return ParseBool(value.asString(), result);
  }

  if (value.isBool()) {
    result = value.asBool();
    return true;
  }

  return false;
}

bool ttv::ParseBool(const ttv::json::Value& root, const char* key, bool& result, bool defaultValue) {
  result = defaultValue;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  }

  if (value.isString()) {
    return ParseBool(value.asString(), result);
  }

  if (value.isBool()) {
    result = value.asBool();
    return true;
  }

  return false;
}

bool ttv::ParseUInt(const ttv::json::Value& root, const char* key, uint64_t& result) {
  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull() || !value.isNumeric()) {
    return false;
  }

  result = value.asUInt();
  return true;
}

bool ttv::ParseUInt(const ttv::json::Value& root, const char* key, uint32_t& result) {
  uint64_t value = 0;
  if (ParseUInt(root, key, value)) {
    result = static_cast<uint32_t>(value);

    return true;
  } else {
    return false;
  }
}

bool ttv::ParseUInt(const ttv::json::Value& root, const char* key, uint64_t& result, uint64_t defaultValue) {
  result = defaultValue;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  }

  if (!value.isNumeric()) {
    return false;
  }

  result = value.asUInt();
  return true;
}

bool ttv::ParseInt(const ttv::json::Value& root, const char* key, int64_t& result) {
  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull() || !value.isNumeric()) {
    return false;
  }

  result = value.asInt();
  return true;
}

bool ttv::ParseInt(const ttv::json::Value& root, const char* key, int64_t& result, int64_t defaultValue) {
  result = defaultValue;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  }

  if (!value.isNumeric()) {
    return false;
  }

  result = value.asInt();
  return true;
}

bool ttv::ParseDouble(const ttv::json::Value& root, const char* key, double& result) {
  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull() || !value.isDouble()) {
    return false;
  }

  result = value.asDouble();
  return true;
}

bool ttv::ParseDouble(const ttv::json::Value& root, const char* key, double& result, double defaultValue) {
  result = defaultValue;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  }

  if (!value.isDouble()) {
    return false;
  }

  result = value.asDouble();
  return true;
}

bool ttv::ParseDocument(const std::string& body, ttv::json::Value& result) {
  return ParseDocument(body.data(), body.data() + body.size(), result);
}

bool ttv::ParseDocument(const char* begin, const char* end, ttv::json::Value& result) {
  ttv::json::Reader reader;

  bool parsed = reader.parse(begin, end, result, false);
  return parsed;
}

bool ttv::ParseDocument(const uint8_t* begin, const uint8_t* end, ttv::json::Value& result) {
  return ParseDocument(reinterpret_cast<const char*>(begin), reinterpret_cast<const char*>(end), result);
}

bool ttv::ParseDocument(const std::vector<char>& body, ttv::json::Value& result) {
  return ParseDocument(body.data(), body.data() + body.size(), result);
}

bool ttv::JsonArrayToVector(const ttv::json::Value& jArray, std::vector<float>& result) {
  for (uint i = 0; i < jArray.size(); ++i) {
    const auto& jEntry = jArray[i];
    float scale;
    if (!ParseFloat(jEntry, scale)) {
      return false;
    }

    result.push_back(scale);
  }

  return true;
}

bool ttv::JsonArrayToVector(const ttv::json::Value& jArray, std::vector<std::string>& result) {
  for (uint i = 0; i < jArray.size(); ++i) {
    const auto& jEntry = jArray[i];
    if (jEntry.isNull() || !jEntry.isString()) {
      return false;
    }

    result.push_back(jEntry.asString());
  }

  return true;
}

bool ttv::ParseUserId(const json::Value& jId, UserId& result) {
  return ParseUnsignedInt32(jId, result);
}

bool ttv::ParseChannelId(const json::Value& jId, ChannelId& result) {
  return ParseUnsignedInt32(jId, result);
}

bool ttv::ParseUInt32(const json::Value& jValue, uint32_t& result) {
  return ParseUnsignedInt32(jValue, result);
}

bool ttv::ParseFloat(const ttv::json::Value& jId, float& result) {
  result = 0;

  if (jId.isNull()) {
    return false;
  } else if (jId.isString()) {
    return ParseNum(jId.asString(), result);
  } else if (jId.isNumeric()) {
    result = static_cast<float>(jId.asDouble());
    return true;
  } else {
    return false;
  }
}

bool ttv::ParseGameId(const json::Value& jId, GameId& result) {
  return ParseUInt32(jId, result);
}

bool ttv::ParseTimestamp(const json::Value& jTimestamp, Timestamp& result) {
  result = 0;

  if (jTimestamp.isNull()) {
    return false;
  } else if (jTimestamp.isString()) {
    return RFC3339TimeToUnixTimestamp(jTimestamp.asString(), result);
  } else if (jTimestamp.isNumeric()) {
    result = static_cast<Timestamp>(jTimestamp.asUInt());
    return true;
  } else {
    return false;
  }
}

bool ttv::ParseTimestamp(const ttv::json::Value& root, const char* key, Timestamp& result) {
  result = 0;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (value.isNull()) {
    return true;
  } else if (value.isString()) {
    return RFC3339TimeToUnixTimestamp(value.asString(), result);
  } else if (value.isNumeric()) {
    result = static_cast<Timestamp>(value.asUInt());
    return true;
  } else {
    return false;
  }
}

bool ttv::ParseColor(const std::string& str, Color& result) {
  result = 0xFF000000;

  if (str != "") {
    // lower case
    std::string lowerColor = str;
    (void)std::transform(lowerColor.begin(), lowerColor.end(), lowerColor.begin(), ::tolower);

    // convert to a hex value
    if (lowerColor[0] == '#') {
      if (sscanf(&lowerColor[1], "%x", &result) == 1) {
        result = result | 0xFF000000;
        return true;
      }
    }
    // regular integer
    else if (lowerColor[0] >= '0' && lowerColor[0] <= '9') {
      if (sscanf(&lowerColor[0], "%u", &result) == 1) {
        result = result | 0xFF000000;
        return true;
      }
    }
  }

  return false;
}

bool ttv::ParseColor(const ttv::json::Value& root, const char* key, Color& result) {
  result = 0;

  if (root.isNull() || !root.isMember(key)) {
    return false;
  }

  const ttv::json::Value& value = root[key];

  if (!value.isString()) {
    return false;
  }

  return ParseColor(value.asString(), result);
}

bool ttv::ParseColor(const ttv::json::Value& root, const char* key, Color& result, Color defaultValue) {
  if (ParseColor(root, key, result)) {
    return true;
  } else {
    result = defaultValue;
    return false;
  }
}

bool ttv::GenerateColorString(Color color, std::string& colorString) {
  std::stringstream stream;
  stream << "#";
  stream << std::hex << color;

  colorString = stream.str();

  return true;
}

ttv::Optional<ttv::json::Value> ttv::FindValueByPath(
  const ttv::json::Value& root, const std::vector<std::string>& keyPath) {
  const ttv::json::Value* current = &root;

  for (const auto& key : keyPath) {
    if (current->isNonNullObject()) {
      if (!current->isMember(key)) {
        // return null optional
        return {};
      }
      current = &current->operator[](key);
    }
  }

  return *current;
}
