/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/module.h"

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/thread.h"

ttv::ModuleBase::ModuleBase() : mState(State::Uninitialized), mLastReportedState(State::Uninitialized) {}

ttv::IModule::State ttv::ModuleBase::GetState() const {
  return mState;
}

TTV_ErrorCode ttv::ModuleBase::Initialize(const InitializeCallback& /*callback*/) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);

  mComponentContainer = std::make_unique<ComponentContainer>();
  TTV_ErrorCode ec = mComponentContainer->Initialize();

  // NOTE: We don't fire the state change here because the derived module needs to set up its state first.  The derived
  // module will change the state and fire the event

  return ec;
}

TTV_ErrorCode ttv::ModuleBase::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  if (mComponentContainer != nullptr) {
    mComponentContainer->Update();
  }

  switch (mState) {
    case State::ShuttingDown: {
      if (CheckShutdown()) {
        CompleteShutdown();
      }

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ModuleBase::Shutdown(const ShutdownCallback& /*callback*/) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mState = State::ShuttingDown;

  NotifyStateChange();

  return TTV_EC_SUCCESS;
}

bool ttv::ModuleBase::CheckShutdown() {
  if (mComponentContainer != nullptr) {
    if (mComponentContainer->GetState() == ComponentContainer::State::Initialized) {
      mComponentContainer->Shutdown();
    }

    if (mComponentContainer->GetState() != ComponentContainer::State::Uninitialized) {
      return false;
    }
  }

  return true;
}

void ttv::ModuleBase::CompleteShutdown() {
  mComponentContainer.reset();

  mState = State::Uninitialized;
  NotifyStateChange();
}

void ttv::ModuleBase::NotifyStateChange() {
  if (mLastReportedState != mState) {
    mLastReportedState = mState;

    mListeners.Invoke([this](std::shared_ptr<IModuleListener> listener) {
      listener->ModuleStateChanged(this, mState, TTV_EC_SUCCESS);
    });

    if (mState == State::Initialized) {
      mInitializeCallbacks.Flush(TTV_EC_SUCCESS);
    } else if (mState == State::Uninitialized) {
      mShutdownCallbacks.Flush(TTV_EC_SUCCESS);
    }
  }
}

void ttv::ShutdownModulesSync(const std::vector<std::shared_ptr<ttv::IModule>>& modules) {
  // Copy the list
  auto list = modules;

  // Shutdown modules in the order they were given
  for (auto iter = list.begin(); iter != list.end();) {
    std::shared_ptr<IModule> module = *iter;

    if (module->GetState() != IModule::State::Uninitialized) {
      std::function<void()> updateFunction = [list]() {
        // Update modules in the order they were given
        for (auto moduleElement : list) {
          moduleElement->Update();
        }
      };

      std::function<bool()> checkForShutdown = [module]() {
        if (module->GetState() == IModule::State::Initialized) {
          module->Shutdown(nullptr);
        }

        return module->GetState() == IModule::State::Uninitialized;
      };

      while (!checkForShutdown()) {
        updateFunction();

        ttv::Sleep(100);
      }
    }

    iter = list.erase(iter);
  }
}
