/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/pollingeventscheduler.h"

#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"

namespace {
const char* kLoggerName = "PollingEventScheduler";
}

ttv::PollingEventScheduler::PollingEventScheduler()
    : mEventQueue("PollingEventScheduler"),
      mName("PollingEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::Running) {}

ttv::PollingEventScheduler::PollingEventScheduler(const std::string& name)
    : mEventQueue("PollingEventScheduler-" + name),
      mName(name + "-PollingEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::Running) {}

ttv::PollingEventScheduler::~PollingEventScheduler() {
  TTV_ASSERT(mState != EventSchedulerState::Running);
  if (mState == EventSchedulerState::Running) {
    CompleteShutdown();
  }
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PollingEventScheduler \"%s\" destroyed", mName.c_str());
}

void ttv::PollingEventScheduler::Update() {
  if (mState == EventSchedulerState::Running) {
    // While EventQueue has tasks available to run, run them
    while (mEventQueue.WaitForEventWithTimeout(0))
      ;
  } else if (mState == EventSchedulerState::ShuttingDown) {
    CompleteShutdown();
  }
}

ttv::Result<ttv::TaskId> ttv::PollingEventScheduler::ScheduleTask(TaskParams&& taskParams) {
  if (mState != EventSchedulerState::Running) {
    return MakeErrorResult(TTV_EC_NOT_INITIALIZED);
  }

  return MakeSuccessResult(mEventQueue.InsertTask(std::move(taskParams)));
}

TTV_ErrorCode ttv::PollingEventScheduler::CancelTask(TaskId taskId) {
  if (mState != EventSchedulerState::Running) {
    return TTV_EC_NOT_INITIALIZED;
  }

  bool removed = mEventQueue.RemoveTask(taskId);

  if (!removed) {
    return TTV_EC_OPERATION_FAILED;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PollingEventScheduler::Shutdown(TaskFunc&& shutdownTask) {
  EventSchedulerState expected = EventSchedulerState::Running;
  if (!mState.compare_exchange_strong(expected, EventSchedulerState::ShuttingDown)) {
    return TTV_EC_NOT_INITIALIZED;
  }

  mShutDownTask = std::move(shutdownTask);

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PollingEventScheduler \"%s\" shutting down", mName.c_str());

  return TTV_EC_SUCCESS;
}

void ttv::PollingEventScheduler::CompleteShutdown() {
  mState = EventSchedulerState::ShutDown;

  mEventQueue.Clear();

  if (mShutDownTask != nullptr) {
    mShutDownTask();
    mShutDownTask = nullptr;
  }

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PollingEventScheduler \"%s\" shutdown", mName.c_str());
}

ttv::IEventScheduler::EventSchedulerState ttv::PollingEventScheduler::GetState() {
  return mState;
}
