/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/profileimagestatus.h"

#include "twitchsdk/core/channel/ichannellistener.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/task/getprofileimageurltask.h"
#include "twitchsdk/core/task/uploadprofileimagetask.h"

namespace {
const uint64_t kUploadTimeoutMilliseconds = 30 * 1000;
const char* kLoggerName = "ProfileImageStatus";
const char* kTopicPrefix = "user-image-update.";
}  // namespace

ttv::ProfileImageStatus::ProfileImageStatus(const std::shared_ptr<User>& user, UserId userId)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + std::to_string(userId)),
      mUserId(userId),
      mUploadInProgress(false) {
  AddTopic(mPubSubTopic);
}

std::string ttv::ProfileImageStatus::GetLoggerName() const {
  return kLoggerName;
}

bool ttv::ProfileImageStatus::CheckShutdown() {
  if (!PubSubComponent::CheckShutdown()) {
    return false;
  }

  if (mUploadInProgress) {
    return false;
  }

  return true;
}

void ttv::ProfileImageStatus::Update() {
  PubSubComponent::Update();

  if (mUploadInProgress && mUploadTimeout.IsTimedOut()) {
    if (mCallback != nullptr) {
      mCallback(TTV_EC_REQUEST_TIMEDOUT, {});
    }
    mUploadInProgress = false;
  }
}

TTV_ErrorCode ttv::ProfileImageStatus::UploadProfileImage(
  const char* imageBuffer, size_t imageSize, const UploadProfileImageCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (mUploadInProgress) {
    return TTV_EC_REQUEST_PENDING;
  }

  mUploadInProgress = true;
  mUploadTimeout.Reset(kUploadTimeoutMilliseconds);

  mCallback = callback;
  std::string image = std::string(imageBuffer, imageSize);

  GetProfileImageUrlTask::Callback profileImageCallback =
    [this, image, callback](
      GetProfileImageUrlTask* source, TTV_ErrorCode ec, const std::shared_ptr<GetProfileImageUrlTask::Result>& result) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec) && !result->uploadId.empty() && !result->uploadUrl.empty()) {
        mUploadId = result->uploadId;

        // Upload the profile image
        UploadProfileImageTask::Callback uploadCallback =
          [this, callback](UploadProfileImageTask* uploadTaskSource, TTV_ErrorCode uploadEc,
            const std::shared_ptr<UploadProfileImageTask::Result>& /*result*/) {
            CompleteTask(uploadTaskSource);

            if (TTV_FAILED(uploadEc)) {
              mUploadInProgress = false;
              if (mCallback != nullptr) {
                mCallback(uploadEc, {});
              }
            }
          };

        std::shared_ptr<UploadProfileImageTask> uploadTask =
          std::make_shared<UploadProfileImageTask>(result->uploadUrl, image, uploadCallback);
        ec = StartTask(uploadTask);

        if (TTV_FAILED(ec)) {
          mUploadInProgress = false;
          if (mCallback != nullptr) {
            mCallback(ec, {});
          }
        }
      } else {
        mUploadInProgress = false;
        if (mCallback != nullptr) {
          mCallback(TTV_EC_API_REQUEST_FAILED, {});
        }
      }
    };

  auto user = mUser.lock();
  std::shared_ptr<GetProfileImageUrlTask> task =
    std::make_shared<GetProfileImageUrlTask>(mUserId, user->GetOAuthToken()->GetToken(), profileImageCallback);
  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    mUploadInProgress = false;
  }

  return ec;
}

void ttv::ProfileImageStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug,
    "ChannelListener (user-image-update.) SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::ProfileImageStatus::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    std::string type;

    if (!ParseString(msg, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    if (type != "user_image_update") {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }

    UserId userId;
    const auto& jUserId = msg["user_id"];
    ParseUserId(jUserId, userId);
    if (userId != mUserId) {
      return;
    }

    const auto& jImageType = msg["image_type"];
    if (jImageType.isNull() || !jImageType.isString()) {
      Log(MessageLevel::Error, "No 'image_type' field, ignoring");
      return;
    }

    if (jImageType.asString() != "profile_image") {
      Log(MessageLevel::Error, "Image type is not 'profile_image'");
      return;
    }

    const auto& jUploadId = msg["upload_id"];
    if (!jUploadId.isNull() && jUploadId.isString()) {
      const std::string& uploadId = jUploadId.asString();

      const auto& jStatus = msg["status"];
      if (!jStatus.isNull() && jStatus.isString()) {
        TTV_ErrorCode ec;
        std::vector<ProfileImage> images;

        const std::string& status = jStatus.asString();
        if (status == "SUCCESS") {
          const auto& jImages = msg["new_image"];
          if (jImages.isNull() || !jImages.isObject()) {
            Log(MessageLevel::Error, "Invalid pub-sub JSON response");
            return;
          }

          for (auto iter = jImages.begin(); iter != jImages.end(); iter++) {
            ProfileImage image;

            const auto& jImage = (*iter);
            if (jImage.isNull() || !jImage.isObject()) {
              Log(MessageLevel::Error, "Invalid pub-sub JSON response");
              return;
            }

            const auto& jWidth = jImage["width"];
            ParseUInt32(jWidth, image.width);

            const auto& jHeight = jImage["height"];
            ParseUInt32(jHeight, image.height);

            const auto& jUrl = jImage["url"];
            if (!jUrl.isNull() && jUrl.isString()) {
              image.url = jUrl.asString();
            }

            const auto& jFormat = jImage["format"];
            if (!jFormat.isNull() && jFormat.isString()) {
              image.format = jFormat.asString();
            }

            images.emplace_back(std::move(image));
          }

          ec = TTV_EC_SUCCESS;
        } else if (status == "IS_IMAGE_VALIDATION_FAILED") {
          ec = TTV_EC_PROFILEIMAGE_IMAGE_VALIDATION_FAILED;
        } else if (status == "IMAGE_FORMAT_VALIDATION_FAILED") {
          ec = TTV_EC_PROFILEIMAGE_FORMAT_VALIDATION_FAILED;
        } else if (status == "FILE_SIZE_VALIDATION_FAILED") {
          ec = TTV_EC_PROFILEIMAGE_SIZE_VALIDATION_FAILED;
        } else if (status == "BACKEND_FAILURE") {
          ec = TTV_EC_PROFILEIMAGE_BACKEND_FAILURE;
        } else {
          ec = TTV_EC_UNKNOWN_ERROR;
        }

        FireListenerAndCallback(uploadId, images, ec);
      }
    }
  }
}

void ttv::ProfileImageStatus::FireListenerAndCallback(
  const std::string& uploadId, const std::vector<ProfileImage>& images, TTV_ErrorCode ec) {
  if (mUploadId == uploadId) {
    mUploadInProgress = false;
    if (mCallback != nullptr) {
      mCallback(ec, images);
    }
    mUploadTimeout.Complete();
  }

  if (TTV_SUCCEEDED(ec) && mListener != nullptr) {
    mListener->ProfileImageUpdated(images);
  }
}
