/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/pubsub/pubsubclient.h"

#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/pubsub/pubsubclientmessages.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

#include <ctime>

// https://twitch.quip.com/ptTfAnm3oa9t

namespace {
const char* kLogger = "PubSubClient";
}

using namespace ttv::pubsub;

namespace {
const uint64_t kGlobalConnectionRetryJitterMilliseconds = 1000;
const uint64_t kGlobalConnectionCounterResetMilliseconds = 1 * 1000 * 60;
const uint64_t kConnectionRetryMaxBackOff = 120 * 1000;  // 2 minutes

bool IsValidTopicName(const std::string& topic) {
  if (topic.empty()) {
    return false;
  }

  for (char character : topic) {
    if (!isalnum(static_cast<unsigned char>(character)) && character != '_' && character != '.' && character != '-') {
      return false;
    }
  }
  return true;
}
}  // namespace

ttv::PubSubClient::Topic::Topic() {}

ttv::PubSubClient::PubSubClient(std::shared_ptr<User> user, std::shared_ptr<SettingRepository> settingRepository)
    : UserComponent(user),
      mSettingRepository(std::move(settingRepository)),
      mConnectionRetryTimer(kConnectionRetryMaxBackOff, kGlobalConnectionRetryJitterMilliseconds),
      mConnectionState(PubSubState::Disconnected),
      mConnectionPreference(ConnectionPreference::OnDemand),
      mConnectionDesired(false) {
  Log(MessageLevel::Debug, "PubSubClient()");
}

ttv::PubSubClient::~PubSubClient() {}

void ttv::PubSubClient::AddListener(std::shared_ptr<IListener> listener) {
  mListeners.AddListener(listener);
}

void ttv::PubSubClient::RemoveListener(std::shared_ptr<IListener> listener) {
  mListeners.RemoveListener(listener);
}

TTV_ErrorCode ttv::PubSubClient::Initialize() {
  Log(MessageLevel::Debug, "Initialize()");

  TTV_ErrorCode ec = UserComponent::Initialize();
  if (TTV_FAILED(ec)) {
    return ec;
  }

  TTV_ASSERT(mState.client == State::Initialized);
  TTV_ASSERT(mThread == nullptr);

  mConnectionListener = std::make_shared<ConnectionListener>(this);

  // Kick off the thread
  ec = ttv::CreateThread(std::bind(&PubSubClient::ThreadProc, this), kLogger, mThread);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mThread != nullptr);

  mThread->Run();

  return TTV_EC_SUCCESS;
}

void ttv::PubSubClient::CompleteShutdown() {
  mThread.reset();
  mConnectionListener.reset();
  mConnection.reset();
  mDyingConnection.reset();
  mListeners.ClearListeners();

  UserComponent::CompleteShutdown();
}

TTV_ErrorCode ttv::PubSubClient::Shutdown() {
  Log(MessageLevel::Debug, "Shutdown()");

  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<ShutdownServerMessage> msg = std::make_shared<ShutdownServerMessage>();
    mOutgoingQueue.push(msg);
  }

  return ec;
}

bool ttv::PubSubClient::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  return mThread == nullptr;
}

std::string ttv::PubSubClient::GetLoggerName() const {
  return kLogger;
}

void ttv::PubSubClient::Update() {
  if (mState.client == State::Uninitialized) {
    return;
  }

  std::shared_ptr<ClientMessage> clientMessage;
  while (mIncomingQueue.try_pop(clientMessage)) {
    switch (clientMessage->type) {
      case ClientMessageType::StateChanged: {
        Log(MessageLevel::Debug, "ClientMessageType::StateChanged");

        std::shared_ptr<StateChangedClientMessage> msg =
          std::static_pointer_cast<StateChangedClientMessage>(clientMessage);
        mConnectionState.client = msg->state;

        for (const auto& listener : msg->listeners) {
          listener->OnStateChanged(this, msg->state, msg->ec);
        }

        break;
      }
      case ClientMessageType::SendMessageResult: {
        Log(MessageLevel::Debug, "ClientMessageType::SendMessageResult");

        std::shared_ptr<SendMessageResultClientMessage> msg =
          std::static_pointer_cast<SendMessageResultClientMessage>(clientMessage);

        if (msg->callback != nullptr) {
          msg->callback(this, msg->ec);
        }

        break;
      }
      case ClientMessageType::MessageReceived: {
        Log(MessageLevel::Debug, "ClientMessageType::MessageReceived");

        std::shared_ptr<MessageReceivedClientMessage> msg =
          std::static_pointer_cast<MessageReceivedClientMessage>(clientMessage);

        for (const auto& listener : msg->listeners) {
          listener->OnTopicMessageReceived(this, msg->topic, msg->data);
        }

        break;
      }
      case ClientMessageType::TopicSubscriptionChanged: {
        Log(MessageLevel::Debug, "ClientMessageType::TopicSubscriptionChanged");

        std::shared_ptr<TopicSubscriptionChangedClientMessage> msg =
          std::static_pointer_cast<TopicSubscriptionChangedClientMessage>(clientMessage);

        for (const auto& listener : msg->listeners) {
          listener->OnTopicSubscribeStateChanged(this, msg->topic, msg->state, msg->ec);
        }

        break;
      }
      case ClientMessageType::TopicListenerRemoved: {
        Log(MessageLevel::Debug, "ClientMessageType::TopicListenerRemoved");

        std::shared_ptr<TopicListenerRemovedClientMessage> msg =
          std::static_pointer_cast<TopicListenerRemovedClientMessage>(clientMessage);

        msg->listener->OnTopicListenerRemoved(this, msg->topic, msg->ec);

        break;
      }
      case ClientMessageType::AuthErrorReceived: {
        Log(MessageLevel::Debug, "ClientMessageType::AuthErrorReceived");

        auto user = mUser.lock();
        if (user != nullptr) {
          std::shared_ptr<AuthErrorReceivedClientMessage> msg =
            std::static_pointer_cast<AuthErrorReceivedClientMessage>(clientMessage);
          user->ReportOAuthTokenInvalid(msg->authToken, msg->ec);
        }

        break;
      }
      case ClientMessageType::ShutdownComplete: {
        Log(MessageLevel::Debug, "ClientMessageType::ShutdownComplete");

        if (mThread != nullptr) {
          mThread->Join();
          mThread.reset();
        }

        // Notify any remaining topic listeners that they have been removed
        for (auto& kvp : mTopics) {
          const std::string& topic = kvp.first;
          kvp.second->listeners.Invoke([this, topic](std::shared_ptr<ITopicListener> listener) {
            listener->OnTopicListenerRemoved(this, topic, TTV_EC_SHUT_DOWN);
          });
        }
        mTopics.clear();

        break;
      }
      default: {
        Log(MessageLevel::Error, "Unhandled client message");
        TTV_ASSERT(false);
        break;
      }
    }
  }

  Component::Update();
}

TTV_ErrorCode ttv::PubSubClient::SetConnectionPreference(ConnectionPreference::Enum preference) {
  Log(MessageLevel::Debug, "SetConnectionPreference(): %d", static_cast<int>(preference));

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto msg = std::make_shared<ConnectionPreferenceServerMessage>(preference);
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::Connect() {
  Log(MessageLevel::Debug, "Connect()");

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto msg = std::make_shared<ConnectServerMessage>();
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::Disconnect() {
  Log(MessageLevel::Debug, "Disconnect()");

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto msg = std::make_shared<DisconnectServerMessage>();
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::SendMessage(const json::Value& jMessage, SendMessageCallback resultCallback) {
  Log(MessageLevel::Debug, "SendMessage()");

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  std::shared_ptr<SendMessageServerMessage> msg = std::make_shared<SendMessageServerMessage>(jMessage, resultCallback);
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::AddTopicListener(const std::string& topic, std::shared_ptr<ITopicListener> listener) {
  Log(MessageLevel::Debug, "AddTopicListener(): %s", topic.c_str());

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!IsValidTopicName(topic)) {
    Log(MessageLevel::Error, "Invalid topic");
    return TTV_EC_PUBSUB_BAD_TOPIC;
  }

  if (listener == nullptr) {
    Log(MessageLevel::Error, "NULL listener");
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  std::shared_ptr<SubscribeToTopicServerMessage> msg =
    std::make_shared<SubscribeToTopicServerMessage>(topic, user->GetOAuthToken(), listener);
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::RemoveTopicListener(
  const std::string& topic, std::shared_ptr<ITopicListener> listener) {
  Log(MessageLevel::Debug, "RemoveTopicListener(): %s", topic.c_str());

  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!IsValidTopicName(topic)) {
    Log(MessageLevel::Error, "Invalid topic");
    return TTV_EC_PUBSUB_BAD_TOPIC;
  }

  if (listener == nullptr) {
    Log(MessageLevel::Error, "NULL listener");
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  std::shared_ptr<UnsubscribeFromTopicServerMessage> msg =
    std::make_shared<UnsubscribeFromTopicServerMessage>(topic, listener);
  mOutgoingQueue.push(msg);

  return TTV_EC_SUCCESS;
}

std::vector<std::string> ttv::PubSubClient::GetSubscribedTopics() const {
  std::vector<std::string> subscribedTopics;

  for (const auto& kvp : mTopics) {
    subscribedTopics.push_back(kvp.first);
  }

  return subscribedTopics;
}

bool ttv::PubSubClient::AnyConnected() const {
  if (MainConnected()) {
    return true;
  }

  if (mDyingConnection != nullptr && mDyingConnection->Connected()) {
    return true;
  }

  return false;
}

bool ttv::PubSubClient::MainConnected() const {
  return mConnection != nullptr && mConnection->Connected();
}

TTV_ErrorCode ttv::PubSubClient::ProcessRequestQueue() {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  std::shared_ptr<ServerMessage> msg;
  while (mOutgoingQueue.try_pop(msg)) {
    switch (msg->type) {
      case ServerMessageType::Connect: {
        ec = ProcessConnect(std::static_pointer_cast<ConnectServerMessage>(msg));
        break;
      }
      case ServerMessageType::Disconnect: {
        ec = ProcessDisconnect(std::static_pointer_cast<DisconnectServerMessage>(msg));
        break;
      }
      case ServerMessageType::SendMessage: {
        ec = ProcessSendMessage(std::static_pointer_cast<SendMessageServerMessage>(msg));
        break;
      }
      case ServerMessageType::SubscribeToTopic: {
        ec = ProcessSubscribeToTopic(std::static_pointer_cast<SubscribeToTopicServerMessage>(msg));
        break;
      }
      case ServerMessageType::UnsubscribeFromTopic: {
        ec = ProcessUnsubscribeFromTopic(std::static_pointer_cast<UnsubscribeFromTopicServerMessage>(msg));
        break;
      }
      case ServerMessageType::ConnectionPreference: {
        ec = ProcessConnectionPreference(std::static_pointer_cast<ConnectionPreferenceServerMessage>(msg));
        break;
      }
      case ServerMessageType::Shutdown: {
        ec = ProcessShutdown(std::static_pointer_cast<ShutdownServerMessage>(msg));
        break;
      }
      default: {
        TTV_ASSERT(false);
        break;
      }
    }
  }

  return ec;
}

void ttv::PubSubClient::SetConnectionState(PubSubState state, TTV_ErrorCode ec) {
  if (mConnectionState.server == state) {
    return;
  }

  mConnectionState.server = state;

  Log(MessageLevel::Debug, "SetConnectionState(): %d", static_cast<int>(state));

  switch (state) {
    case PubSubState::Disconnected: {
      mConnectionRetryTimer.ClearGlobalReset();

      break;
    }
    case PubSubState::Connecting: {
      mConnectionRetryTimer.ClearGlobalReset();

      break;
    }
    case PubSubState::Connected: {
      mConnectionRetryTimer.StartGlobalReset(kGlobalConnectionCounterResetMilliseconds);

      SyncTopicSubscriptions();

      break;
    }
    case PubSubState::Disconnecting: {
      break;
    }
  }

  // Pass the state change to the client
  std::shared_ptr<StateChangedClientMessage> msg = std::make_shared<StateChangedClientMessage>(state, ec);
  mListeners.CaptureListeners(msg->listeners);
  mIncomingQueue.push(msg);
}

TTV_ErrorCode ttv::PubSubClient::ProcessConnectionPreference(
  std::shared_ptr<pubsub::ConnectionPreferenceServerMessage> msg) {
  if (mConnectionPreference == msg->preference) {
    return TTV_EC_SUCCESS;
  }

  mConnectionPreference = msg->preference;

  // Connect or disconnect if we're managing the connection
  if (mConnectionPreference == ConnectionPreference::OnDemand) {
    if (mTopics.empty()) {
      PerformDisconnect();
    } else {
      if (!MainConnected()) {
        ScheduleConnect(TTV_EC_SUCCESS);
      }
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::ProcessConnect(std::shared_ptr<ConnectServerMessage> /*msg*/) {
  Log(MessageLevel::Debug, "ProcessConnect()");

  mConnectionDesired = true;
  mConnectionPreference = ConnectionPreference::Manual;

  return ScheduleConnect(TTV_EC_SUCCESS);
}

TTV_ErrorCode ttv::PubSubClient::ProcessDisconnect(std::shared_ptr<DisconnectServerMessage> /*msg*/) {
  Log(MessageLevel::Debug, "ProcessDisconnect()");

  mConnectionDesired = false;
  mConnectionPreference = ConnectionPreference::Manual;

  return PerformDisconnect();
}

TTV_ErrorCode ttv::PubSubClient::ProcessSendMessage(std::shared_ptr<SendMessageServerMessage> msg) {
  Log(MessageLevel::Debug, "ProcessSendMessage()");

  if (!AnyConnected()) {
    Log(MessageLevel::Debug, "Not connected so message not sent");
    return TTV_EC_SOCKET_ENOTCONN;
  }

  std::string text = mJsonWriter.write(msg->data);
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (mConnection != nullptr) {
    ec = mConnection->Send(text);
  }

  if (TTV_FAILED(ec) && mDyingConnection != nullptr) {
    Log(MessageLevel::Debug, "No main connection, trying to send on the dying connection");
    ec = mDyingConnection->Send(text);
  }

  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to send pubsub message: %s", ErrorToString(ec));
  }

  return ec;
}

TTV_ErrorCode ttv::PubSubClient::ScheduleConnect(TTV_ErrorCode ec) {
  if (mConnection != nullptr) {
    mConnection->Disconnect();
  }

  mConnectionRetryTimer.ScheduleNextRetry();

  SetConnectionState(PubSubState::Connecting, ec);

  return TTV_EC_SUCCESS;
}

void ttv::PubSubClient::PerformReconnect() {
  Log(MessageLevel::Debug, "PerformReconnect(): Scheduling reconnect which might result in overlapping connections");

  // The edge host is going to restart in 15 seconds so we need to spin up another connection,
  // resubscribe to all topics and get the new connection up to speed before the previous connection
  // is cut off on us.

  if (mDyingConnection == nullptr) {
    // Move the current socket over to the dying socket
    if (mConnection != nullptr) {
      Log(MessageLevel::Debug, "PerformReconnect(): Moving connection %u is now dying",
        mConnection->GetConnectionIndex());

      mDyingConnection = mConnection;
      mConnection.reset();
    }
  }
  // Already in a reconnecting state
  else {
    if (mConnection != nullptr) {
      // Keep a reference around so the state callback doesn't free it
      auto connection = mDyingConnection;

      Log(MessageLevel::Debug,
        "PerformReconnect(): Already in reconnecting state, killing %connection u and moving %u to dying",
        mDyingConnection->GetConnectionIndex(), mConnection->GetConnectionIndex());

      mDyingConnection->Disconnect();

      mDyingConnection = mConnection;
      mConnection.reset();
    }
  }

  // Connect to a new edge server ASAP
  ScheduleConnect(TTV_EC_SOCKET_TRY_AGAIN);
}

void ttv::PubSubClient::UpdateTopicSubscription(const std::string& topicName) {
  // Apply changes if connected
  if (MainConnected()) {
    auto iter = mTopics.find(topicName);
    if (iter == mTopics.end()) {
      Log(MessageLevel::Error, "Couldn't find topic: %s", topicName.c_str());
      return;
    }

    auto topic = iter->second;

    PubSubClientConnection::TopicSubscriptionState::Enum state = mConnection->GetTopicState(topicName);
    bool shouldBeSubscribed = !topic->listeners.Empty();

    // We were waiting for a previous sub change to complete before applying another one
    if (state == PubSubClientConnection::TopicSubscriptionState::Subscribed) {
      if (!shouldBeSubscribed) {
        Log(MessageLevel::Debug, "Topic out of sync, unsubscribing to topic: %s", topicName.c_str());
        auto connection = mConnection;
        connection->Unlisten(topicName);
      }
    } else if (state == PubSubClientConnection::TopicSubscriptionState::Unsubscribed) {
      auto user = mUser.lock();
      if (user != nullptr) {
        if (shouldBeSubscribed) {
          Log(MessageLevel::Debug, "Topic out of sync, subscribing to topic: %s", topicName.c_str());
          auto connection = mConnection;
          connection->Listen(topicName, user);
        } else {
          mTopics.erase(iter);
        }
      }
    }
  }

  // Perform automatic connection/disconnection if necessary
  if (mConnectionPreference == ConnectionPreference::OnDemand && mState.server < State::ShuttingDown) {
    bool connected = MainConnected();
    mConnectionDesired = !mTopics.empty();

    if (!mConnectionDesired && connected) {
      Log(MessageLevel::Debug, "Disconnecting due to OnDemand preference");
      PerformDisconnect();
    } else if (mConnectionDesired && !connected) {
      Log(MessageLevel::Debug, "Connecting due to OnDemand preference");
      if (!mConnectionRetryTimer.IsRetrySet()) {
        ScheduleConnect(TTV_EC_SUCCESS);
      }
    }
  }
}

ttv::PubSubClient::SubscribeState::Enum ttv::PubSubClient::GetEffectiveTopicState(const std::string& topic) {
  if (mConnection != nullptr) {
    if (mConnection->GetTopicState(topic) == PubSubClientConnection::TopicSubscriptionState::Subscribed) {
      return SubscribeState::Subscribed;
    }
  }

  if (mDyingConnection != nullptr) {
    if (mDyingConnection->GetTopicState(topic) == PubSubClientConnection::TopicSubscriptionState::Subscribed) {
      return SubscribeState::Subscribed;
    }
  }

  return SubscribeState::Unsubscribed;
}

TTV_ErrorCode ttv::PubSubClient::ProcessSubscribeToTopic(std::shared_ptr<SubscribeToTopicServerMessage> msg) {
  Log(MessageLevel::Debug, "ProcessSubscribeToTopic(): %s", msg->topic.c_str());

  std::shared_ptr<Topic> topic;
  std::string name = msg->topic;

  auto iter = mTopics.find(name);
  if (iter == mTopics.end()) {
    Log(MessageLevel::Debug, "Topic does not exist yet, creating: %s", msg->topic.c_str());

    topic = std::make_shared<Topic>();
    topic->topic = name;

    mTopics[name] = topic;
  } else {
    Log(MessageLevel::Debug, "Topic already exists: %s", msg->topic.c_str());

    topic = iter->second;
  }

  topic->listeners.AddListener(msg->listener);

  // Report the current subscription state to the new listener
  SubscribeState::Enum state = GetEffectiveTopicState(name);
  std::shared_ptr<TopicSubscriptionChangedClientMessage> clientMessage =
    std::make_shared<TopicSubscriptionChangedClientMessage>(name, state, TTV_EC_SUCCESS);
  clientMessage->listeners.push_back(msg->listener);
  mIncomingQueue.push(clientMessage);

  // Kick off subscription on the connection if needed
  UpdateTopicSubscription(name);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::ProcessUnsubscribeFromTopic(std::shared_ptr<UnsubscribeFromTopicServerMessage> msg) {
  Log(MessageLevel::Debug, "ProcessUnsubscribeFromTopic(): %s", msg->topic.c_str());

  std::shared_ptr<Topic> topic;
  std::string name = msg->topic;

  auto iter = mTopics.find(msg->topic);
  if (iter != mTopics.end()) {
    topic = iter->second;
    topic->listeners.RemoveListener(msg->listener);

    UpdateTopicSubscription(name);
  }

  // Notify the listener that it was removed, even if it wasn't subbed
  std::shared_ptr<TopicListenerRemovedClientMessage> removedMessage =
    std::make_shared<TopicListenerRemovedClientMessage>(msg->listener, name, TTV_EC_SUCCESS);
  mIncomingQueue.push(removedMessage);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::ProcessShutdown(std::shared_ptr<pubsub::ShutdownServerMessage> /*msg*/) {
  Log(MessageLevel::Debug, "ProcessShutdown()");

  if (mState.server >= State::ShuttingDown) {
    return TTV_EC_SUCCESS;
  }

  mConnectionRetryTimer.Clear();

  SetServerState(State::ShuttingDown);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::PerformDisconnect() {
  Log(MessageLevel::Debug, "PerformDisconnect()");

  if (mConnection != nullptr) {
    auto connection = mConnection;
    mConnection->Disconnect();
    mConnection.reset();
  }

  if (mDyingConnection != nullptr) {
    auto connection = mDyingConnection;
    mDyingConnection->Disconnect();
    mDyingConnection.reset();
  }

  SetConnectionState(PubSubState::Disconnected, TTV_EC_SUCCESS);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::PubSubClient::SyncTopicSubscriptions() {
  Log(MessageLevel::Debug, "SyncTopicSubscriptions()");

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  // Only sync on the main connection
  if (!MainConnected()) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // Now sync all topic subscriptions
  for (const auto& kvp : mTopics) {
    auto topic = kvp.second;
    if (!topic->listeners.Empty()) {
      auto connection = mConnection;
      ec = connection->Listen(kvp.first, user);
      if (TTV_FAILED(ec)) {
        // The only error that could happen here is a disconnect and we'll get that error when polling the socket
        Log(MessageLevel::Error, "Failed to listen to topic during sync");
        break;
      }
    }
  }

  return ec;
}

TTV_ErrorCode ttv::PubSubClient::AttemptConnection() {
  Log(MessageLevel::Debug, "AttemptConnection()");

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  mConnection = std::make_shared<PubSubClientConnection>(user, mSettingRepository.get());
  mConnection->AddListener(mConnectionListener);

  TTV_ErrorCode ec = mConnection->Connect();

  if (TTV_SUCCEEDED(ec)) {
    SetConnectionState(PubSubState::Connected, ec);
  } else {
    Log(MessageLevel::Error, "Failed to connect");

    mConnection.reset();

    ScheduleConnect(ec);
  }

  return ec;
}

void ttv::PubSubClient::ThreadProc() {
  while (mState.server < State::ShuttingDown) {
    ProcessRequestQueue();

    mConnectionRetryTimer.CheckGlobalReset();

    // Check if it's time to connect
    if (mConnectionDesired) {
      bool connected = mConnection != nullptr && mConnection->Connected();

      if (!connected && mConnectionRetryTimer.CheckNextRetry()) {
        AttemptConnection();
      }
    }

    if (mDyingConnection != nullptr) {
      // Keep a reference here since a disconnect event from the connection will clear the stored reference
      auto connection = mDyingConnection;
      connection->Update();
      connection->PollSocket();
    }

    if (mConnection != nullptr) {
      // Keep a reference here since a disconnect event from the connection will clear the stored reference
      auto connection = mConnection;
      connection->Update();
      connection->PollSocket();
    }

    Sleep(250);
  }

  Log(MessageLevel::Debug, "Shutting down");

  // Cleanup
  PerformDisconnect();

  SetServerState(State::Uninitialized);

  // Notify the main thread that shutdown is complete
  std::shared_ptr<ShutdownCompleteClientMessage> clientMessage = std::make_shared<ShutdownCompleteClientMessage>();
  mIncomingQueue.push(clientMessage);

  Log(MessageLevel::Debug, "Thread finished");
}

void ttv::PubSubClient::OnReconnectReceived(PubSubClientConnection* connection) {
  // We don't care about changes on the dying connection
  if (mConnection.get() != connection) {
    Log(MessageLevel::Debug, "OnReconnectReceived() from dying connection %u, ignoring",
      connection->GetConnectionIndex());
    return;
  }

  Log(MessageLevel::Debug, "OnReconnectReceived() from main connection %u", connection->GetConnectionIndex());

  PerformReconnect();
}

void ttv::PubSubClient::OnConnectionStateChanged(
  PubSubClientConnection* connection, PubSubState state, TTV_ErrorCode ec) {
  if (mConnection.get() == connection) {
    Log(MessageLevel::Debug, "OnConnectionStateChanged() from main connection %u, state: %d",
      connection->GetConnectionIndex(), static_cast<int>(state));

    if (state == PubSubState::Disconnected) {
      Log(MessageLevel::Debug, "OnConnectionStateChanged() discarding main connection %u",
        connection->GetConnectionIndex());

      // NOTE: This connection is on the stack so we keep a ref in the code that calls into PubSubClientConnection
      mConnection.reset();

      // We want to report a reconnecting state if we're supposed to be connected
      if (mConnectionDesired) {
        state = PubSubState::Connecting;
      }
    }

    SetConnectionState(state, ec);
  } else if (mDyingConnection.get() == connection) {
    Log(MessageLevel::Debug, "OnConnectionStateChanged() from dying connection %u, state: %d",
      connection->GetConnectionIndex(), static_cast<int>(state));

    SetConnectionState(state, ec);

    if (state == PubSubState::Disconnected) {
      Log(MessageLevel::Debug, "OnConnectionStateChanged() discarding dying connection %u",
        connection->GetConnectionIndex());

      // NOTE: This connection is on the stack so we keep a ref in the code that calls into PubSubClientConnection
      mDyingConnection.reset();
    }
  }
}

void ttv::PubSubClient::OnTopicSubscriptionChanged(PubSubClientConnection* connection, const std::string& topicName,
  PubSubClientConnection::TopicSubscriptionState::Enum state, TTV_ErrorCode ec) {
  // We don't care about changes on the dying connection
  if (mConnection.get() != connection) {
    Log(MessageLevel::Debug, "OnTopicSubscriptionChanged(): %s %d from dying connection %u, discarding",
      topicName.c_str(), static_cast<int>(state), connection->GetConnectionIndex());

    return;
  }

  Log(MessageLevel::Debug, "OnTopicSubscriptionChanged(): %s %d from main connection %u", topicName.c_str(),
    static_cast<int>(state), connection->GetConnectionIndex());

  // Report the change to clients
  auto iter = mTopics.find(topicName);
  if (iter != mTopics.end()) {
    auto topic = iter->second;

    std::shared_ptr<TopicSubscriptionChangedClientMessage> clientMessage =
      std::make_shared<TopicSubscriptionChangedClientMessage>(topicName, GetEffectiveTopicState(topicName), ec);
    topic->listeners.CaptureListeners(clientMessage->listeners);
    mIncomingQueue.push(clientMessage);
  }

  // Trigger subscriptions changes if needed
  UpdateTopicSubscription(topicName);
}

void ttv::PubSubClient::OnTopicMessageReceived(
  PubSubClientConnection* connection, const std::string& topicName, const json::Value& message) {
  if (mConnection.get() != connection) {
    Log(MessageLevel::Debug, "OnTopicMessageReceived(): %s from dying connection %u", topicName.c_str(),
      connection->GetConnectionIndex());
  } else {
    Log(MessageLevel::Debug, "OnTopicMessageReceived(): %s from main connection %u", topicName.c_str(),
      connection->GetConnectionIndex());
  }

  // Pass the message on to clients
  auto iter = mTopics.find(topicName);
  if (iter != mTopics.end()) {
    auto topic = iter->second;

    std::shared_ptr<MessageReceivedClientMessage> clientMessage =
      std::make_shared<MessageReceivedClientMessage>(topicName, message);
    topic->listeners.CaptureListeners(clientMessage->listeners);
    mIncomingQueue.push(clientMessage);
  }
}

void ttv::PubSubClient::OnPongTimeout(PubSubClientConnection* connection) {
  // If the pong timer goes off then we need to reconnect to the server
  if (mConnection.get() != connection) {
    Log(MessageLevel::Debug, "OnPongTimeout(): From dying connection %u, ignoring", connection->GetConnectionIndex());
  } else {
    Log(MessageLevel::Debug, "OnPongTimeout(): From main connection %u", connection->GetConnectionIndex());
    PerformReconnect();
  }
}

void ttv::PubSubClient::OnAuthenticationEror(
  ttv::PubSubClientConnection* /*connection*/, TTV_ErrorCode ec, const std::shared_ptr<const OAuthToken>& authToken) {
  std::shared_ptr<AuthErrorReceivedClientMessage> clientMessage =
    std::make_shared<AuthErrorReceivedClientMessage>(ec, authToken);
  mIncomingQueue.push(clientMessage);
}

ttv::PubSubClient::ConnectionListener::ConnectionListener(PubSubClient* pubsub) : mPubSub(pubsub) {}

void ttv::PubSubClient::ConnectionListener::OnReconnectReceived(PubSubClientConnection* connection) {
  mPubSub->OnReconnectReceived(connection);
}

void ttv::PubSubClient::ConnectionListener::OnConnectionStateChanged(
  PubSubClientConnection* connection, PubSubState state, TTV_ErrorCode ec) {
  mPubSub->OnConnectionStateChanged(connection, state, ec);
}

void ttv::PubSubClient::ConnectionListener::OnTopicSubscriptionChanged(PubSubClientConnection* connection,
  const std::string& topicName, PubSubClientConnection::TopicSubscriptionState::Enum state, TTV_ErrorCode ec) {
  mPubSub->OnTopicSubscriptionChanged(connection, topicName, state, ec);
}

void ttv::PubSubClient::ConnectionListener::OnTopicMessageReceived(
  PubSubClientConnection* connection, const std::string& topic, const json::Value& message) {
  mPubSub->OnTopicMessageReceived(connection, topic, message);
}

void ttv::PubSubClient::ConnectionListener::OnPongTimeout(PubSubClientConnection* connection) {
  mPubSub->OnPongTimeout(connection);
}

void ttv::PubSubClient::ConnectionListener::OnAuthenticationError(
  ttv::PubSubClientConnection* connection, TTV_ErrorCode ec, const std::shared_ptr<const OAuthToken>& authToken) {
  mPubSub->OnAuthenticationEror(connection, ec, authToken);
}

//// PubSubTopicListenerHelper
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

ttv::PubSubTopicListenerHelper::PubSubTopicListenerHelper(
  std::shared_ptr<PubSubClient> pubsub, std::shared_ptr<PubSubClient::ITopicListener> listener)
    : mPubSub(pubsub), mListener(listener), mState(State::Initialized) {}

ttv::PubSubTopicListenerHelper::~PubSubTopicListenerHelper() {
  TTV_ASSERT(mSubscriptionStates.empty());
}

ttv::PubSubClient::SubscribeState::Enum ttv::PubSubTopicListenerHelper::GetSubscriptionState(const std::string& topic) {
  auto iter = mSubscriptionStates.find(topic);
  if (iter == mSubscriptionStates.end()) {
    return PubSubClient::SubscribeState::Unsubscribed;
  } else {
    return iter->second.state;
  }
}

TTV_ErrorCode ttv::PubSubTopicListenerHelper::Subscribe(const std::string& topic) {
  if (mPubSub == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto state = GetSubscriptionState(topic);
  if (state == PubSubClient::SubscribeState::Subscribed) {
    return TTV_EC_SUCCESS;
  }

  TTV_ErrorCode ec = mPubSub->AddTopicListener(topic, shared_from_this());

  if (TTV_SUCCEEDED(ec)) {
    auto iter = mSubscriptionStates.find(topic);
    if (iter == mSubscriptionStates.end()) {
      Entry entry;
      entry.state = PubSubClient::SubscribeState::Unsubscribed;
      entry.subscriptionDesired = true;

      mSubscriptionStates[topic] = entry;
      iter = mSubscriptionStates.find(topic);
    } else {
      iter->second.state = PubSubClient::SubscribeState::Unsubscribed;
      iter->second.subscriptionDesired = true;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::PubSubTopicListenerHelper::Unsubscribe(const std::string& topic) {
  if (mPubSub == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto iter = mSubscriptionStates.find(topic);
  if (iter == mSubscriptionStates.end()) {
    return TTV_EC_SUCCESS;
  }

  iter->second.subscriptionDesired = false;

  if (iter->second.state == PubSubClient::SubscribeState::Unsubscribed) {
    return TTV_EC_SUCCESS;
  }

  TTV_ErrorCode ec = mPubSub->RemoveTopicListener(topic, shared_from_this());

  if (TTV_SUCCEEDED(ec)) {
    // NOTE: The actual removal confirmation will come in the listener callback
  }

  return ec;
}

bool ttv::PubSubTopicListenerHelper::ContainsTopic(const std::string& topic) const {
  return mSubscriptionStates.find(topic) != mSubscriptionStates.end();
}

void ttv::PubSubTopicListenerHelper::Shutdown() {
  if (mState != State::Initialized) {
    return;
  }

  if (!mSubscriptionStates.empty()) {
    mState = State::ShuttingDown;

    for (auto& kvp : mSubscriptionStates) {
      mPubSub->RemoveTopicListener(kvp.first, shared_from_this());
    }
  } else {
    mState = State::Shutdown;
  }
}

void ttv::PubSubTopicListenerHelper::OnTopicSubscribeStateChanged(
  PubSubClient* source, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  // Avoid redundant updates
  auto localState = GetSubscriptionState(topic);
  if (localState == state) {
    return;
  }

  auto iter = mSubscriptionStates.find(topic);
  assert(iter != mSubscriptionStates.end());
  iter->second.state = state;

  auto listener = mListener.lock();
  if (listener != nullptr) {
    listener->OnTopicSubscribeStateChanged(source, topic, state, ec);
  }
}

void ttv::PubSubTopicListenerHelper::OnTopicMessageReceived(
  PubSubClient* source, const std::string& topic, const json::Value& msg) {
  auto listener = mListener.lock();
  if (listener != nullptr) {
    listener->OnTopicMessageReceived(source, topic, msg);
  }
}

void ttv::PubSubTopicListenerHelper::OnTopicListenerRemoved(
  PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) {
  auto iter = mSubscriptionStates.find(topic);
  if (iter != mSubscriptionStates.end()) {
    mSubscriptionStates.erase(iter);
  }

  auto listener = mListener.lock();
  if (listener != nullptr) {
    listener->OnTopicListenerRemoved(source, topic, ec);
  }

  // Check to see if shutdown is complete
  if (mState == State::ShuttingDown && mSubscriptionStates.empty()) {
    mState = State::Shutdown;
  }
}
