/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/pubsub/pubsubcomponent.h"

ttv::PubSubComponentBase::PubSubComponentBase(const std::shared_ptr<User>& user) : UserComponent(user) {}

TTV_ErrorCode ttv::PubSubComponentBase::Initialize() {
  TTV_ErrorCode ec = UserComponent::Initialize();
  if (TTV_FAILED(ec)) {
    return ec;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  mPubSub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (mPubSub != nullptr) {
    mPubSubTopicListener = std::make_shared<PubSubTopicListener>(this);
    mPubSubTopicListenerHelper = std::make_shared<PubSubTopicListenerHelper>(mPubSub, mPubSubTopicListener);

    SubscribeTopics();
  } else {
    return TTV_EC_NEED_TO_LOGIN;
  }

  return TTV_EC_SUCCESS;
}

void ttv::PubSubComponentBase::AddTopic(const std::string& topic) {
  mPubSubTopics.emplace_back(topic);
}

TTV_ErrorCode ttv::PubSubComponentBase::SubscribeTopics() {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (mPubSubTopicListenerHelper == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  } else if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  for (const std::string& topic : mPubSubTopics) {
    if (!mPubSubTopicListenerHelper->ContainsTopic(topic)) {
      ec = mPubSubTopicListenerHelper->Subscribe(topic);

      if (TTV_FAILED(ec)) {
        Log(MessageLevel::Error, "Error subscribing to pubsub topic %s - %s", topic.c_str(), ErrorToString(ec));
      }
    }
  }

  return ec;
}

TTV_ErrorCode ttv::PubSubComponentBase::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mPubSubTopicListenerHelper != nullptr) {
      mPubSubTopicListenerHelper->Shutdown();
    }
  }

  return ec;
}

bool ttv::PubSubComponentBase::CheckShutdown() {
  if (!UserComponent::CheckShutdown()) {
    return false;
  }

  // Make sure unsubscribed from pubsub topics
  if (mPubSubTopicListenerHelper != nullptr &&
      mPubSubTopicListenerHelper->GetState() != PubSubTopicListenerHelper::State::Shutdown) {
    return false;
  }

  return true;
}

void ttv::PubSubComponentBase::CompleteShutdown() {
  UserComponent::CompleteShutdown();

  mPubSub.reset();
  mPubSubTopicListener.reset();
  mPubSubTopicListenerHelper.reset();
}

TTV_ErrorCode ttv::PubSubComponentBase::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::PubSubComponentBase::OnTopicListenerRemoved(const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {
  // NOTE: The listener is only removed if we requested it to be removed or pubsub is shutting down
}

void ttv::PubSubComponentBase::PubSubTopicListener::OnTopicSubscribeStateChanged(
  ttv::PubSubClient* /*source*/, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  mOwner->OnTopicSubscribeStateChanged(topic, state, ec);
}

void ttv::PubSubComponentBase::PubSubTopicListener::OnTopicMessageReceived(
  ttv::PubSubClient* /*source*/, const std::string& topic, const json::Value& msg) {
  mOwner->OnTopicMessageReceived(topic, msg);
}

void ttv::PubSubComponentBase::PubSubTopicListener::OnTopicListenerRemoved(
  ttv::PubSubClient* /*source*/, const std::string& topic, TTV_ErrorCode ec) {
  mOwner->OnTopicListenerRemoved(topic, ec);
}
