/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/sockettracker.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/systemclock.h"

// Destructor is defined here so we can forward-declare IMutex in the header
ttv::SocketTracker::~SocketTracker() = default;

void ttv::SocketTracker::AddSendInfo(uint32_t byteCount, uint64_t sentTimestamp, uint64_t blockTime) {
  SendEntry entry;
  entry.bytesSent = byteCount;
  entry.sentTimestamp = sentTimestamp;
  entry.blockTime = blockTime;

  uint64_t now = GetSystemClockTime();
  uint64_t windowCutoff = 60 * GetSystemClockFrequency();  // 60 seconds

  ttv::AutoMutex lock(mMutex.get());
  mSendEntries.push_front(entry);

  if (now - mSendEntries.back().sentTimestamp > windowCutoff) {
    mSendEntries.pop_back();
  }
}

void ttv::SocketTracker::Reset() {
  ttv::AutoMutex lock(mMutex.get());
  mSendEntries.clear();
}

TTV_ErrorCode ttv::SocketTracker::GetAverageOutgoingRate(
  uint64_t measurementWindowMilliseconds, uint64_t& bitsPerSecond) const {
  uint64_t totalBytesSent = 0;
  uint64_t oldestTimestamp = 0;

  uint64_t now = GetSystemClockTime();
  uint64_t windowCutoff = now - (measurementWindowMilliseconds * GetSystemClockFrequency() / 1000);

  {
    ttv::AutoMutex lock(mMutex.get());
    for (const auto& entry : mSendEntries) {
      if (entry.sentTimestamp < windowCutoff) {
        break;
      }
      totalBytesSent += entry.bytesSent;
      oldestTimestamp = entry.sentTimestamp;
    }
  }

  if (oldestTimestamp == 0) {
    return TTV_EC_NOT_AVAILABLE;
  } else {
    bitsPerSecond = (GetSystemClockFrequency() * totalBytesSent * 8) / (now - oldestTimestamp);
    return TTV_EC_SUCCESS;
  }
}

TTV_ErrorCode ttv::SocketTracker::GetEstimatedCongestionLevel(
  uint64_t measurementWindowMilliseconds, double& congestionLevel) const {
  uint64_t totalCongestionTime = 0;
  uint64_t oldestTimestamp = 0;

  uint64_t now = GetSystemClockTime();
  uint64_t windowCutoff = now - (measurementWindowMilliseconds * GetSystemClockFrequency() / 1000);

  {
    ttv::AutoMutex lock(mMutex.get());
    for (const auto& entry : mSendEntries) {
      if (entry.sentTimestamp < windowCutoff) {
        break;
      }
      totalCongestionTime += entry.blockTime;
      oldestTimestamp = entry.sentTimestamp;
    }
  }

  if (oldestTimestamp == 0) {
    return TTV_EC_NOT_AVAILABLE;
  } else {
    congestionLevel = static_cast<double>(totalCongestionTime) / static_cast<double>(now - oldestTimestamp);
  }

  return TTV_EC_SUCCESS;
}

ttv::SocketTracker::SocketTracker() {
  TTV_ErrorCode ret = ttv::CreateMutex(mMutex, "SocketTracker");
  ASSERT_ON_ERROR(ret);
}
