/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/stringutilities.h"

#include "twitchsdk/core/random.h"
#include "twitchsdk/core/systemclock.h"

#include <array>
#include <chrono>
#include <iomanip>
#include <limits>
#include <random>
#include <sstream>

namespace {
uint32_t CountDays(int32_t y, int32_t m, int32_t d) {
  if (m <= 2) {
    y--;
  }

  int32_t era = (y >= 0 ? y : y - 399) / 400;
  int32_t yoe = y - era * 400;                                   // [0, 399]
  int32_t doy = (153 * (m + (m > 2 ? -3 : 9)) + 2) / 5 + d - 1;  // [0, 365]
  int32_t doe = yoe * 365 + yoe / 4 - yoe / 100 + doy;           // [0, 146096]

  return static_cast<uint32_t>(era * 146097 + doe - 719468);
}

bool RFC3339TimeToTimePoint(const std::string& str, std::chrono::system_clock::time_point& result) {
  // using days = std::chrono::duration<int, std::ratio_multiply<std::ratio<24>, std::chrono::hours::period>>;

  using namespace std::chrono;

  std::istringstream is(str);
  const std::time_get<char>& tg = std::use_facet<std::time_get<char>>(is.getloc());

  try {
    uint32_t base = 0;
    uint32_t offset = 0;

    // Read YYYY:MM:DT:HH:MM:SS
    if (!is.eof()) {
      const char patternYMDHMS[] = "%Y-%m-%dT%H:%M:%S";

      std::tm baseTime;
      std::ios_base::iostate err = std::ios_base::goodbit;
      tg.get(is, nullptr, is, err, &baseTime, std::begin(patternYMDHMS), std::end(patternYMDHMS) - 1);

      if (err != std::ios_base::goodbit) {
        return false;
      }

      base = CountDays(baseTime.tm_year + 1900, baseTime.tm_mon + 1, baseTime.tm_mday) * 24 * 60 * 60 +
             static_cast<uint32_t>(baseTime.tm_hour) * 60 * 60 + static_cast<uint32_t>(baseTime.tm_min) * 60 +
             static_cast<uint32_t>(baseTime.tm_sec) * 1;
    }

    // Check if there is a time offset from UTC specified
    char sign = '\0';
    if (!is.eof()) {
      is.get(sign);
      if (is.rdstate() != std::ios_base::goodbit) {
        return false;
      }

      // Throw out fractional seconds
      if (sign == '.') {
        uint32_t frac = 0;
        is >> frac;

        // Try to get the sign again
        if (!is.eof()) {
          is.get(sign);
          if (is.rdstate() != std::ios_base::goodbit) {
            return false;
          }
        }
      }
    }

    // Handle specific time shift from UTC
    if (sign == '+' || sign == '-') {
      std::tm offsetTime;
      const char patternHM[] = "%H:%M";

      std::ios_base::iostate err = std::ios_base::goodbit;
      tg.get(is, nullptr, is, err, &offsetTime, std::begin(patternHM), std::end(patternHM) - 1);

      if (!(err & std::ios_base::failbit)) {
        offset =
          static_cast<uint32_t>((sign == '+' ? 1 : -1) * ((offsetTime.tm_hour * 3600) + (offsetTime.tm_min * 60)));
      }
    }
    // UTC
    else if (sign == 'Z') {
      offset = 0;
    }

    result = system_clock::from_time_t(static_cast<time_t>(base - offset));
  } catch (...) {
    // Eat any errors
    return false;
  }

  return true;
}

inline void PrintByte(std::ostringstream& stream, uint8_t byte) {
  stream << std::setw(2) << std::setfill('0') << +byte;
}
}  // namespace

bool ttv::IsValidChannelName(const std::string& channelName) {
  return IsValidUserName(channelName);
}

bool ttv::IsValidUserName(const std::string& userName) {
  if (userName.empty()) {
    return false;
  }

  for (char character : userName) {
    if (!isalnum(static_cast<unsigned char>(character)) && (character != '_')) {
      return false;
    }
  }
  return true;
}

bool ttv::IsValidOAuthToken(const std::string& oauthToken) {
  return oauthToken != "";
}

int ttv::IsWhitespace(int ch) {
  return ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n';
}

std::string ttv::ToLowerCase(const std::string& str) {
  std::string lower = str;
  (void)std::transform(lower.begin(), lower.end(), lower.begin(), ::tolower);
  return lower;
}

void ttv::Trim(std::string& str) {
  // Trim leading whitespace
  while (!str.empty() && IsWhitespace(str[0])) {
    str.erase(str.begin());
  }

  // Trim trailing whitespace
  while (!str.empty() && IsWhitespace(str[str.size() - 1])) {
    str.erase(str.end() - 1);
  }
}

void ttv::Split(const std::string& str, std::vector<std::string>& result, char sep, bool includeEmpty) {
  unsigned start = 0;
  unsigned end = 0;

  for (;;) {
    if (end == str.size() || str[end] == sep) {
      if (end > start || includeEmpty) {
        result.push_back(str.substr(start, end - start));
      }

      if (end == str.size()) {
        break;
      }

      ++end;
      start = end;
    } else {
      ++end;
    }
  }
}

void ttv::Split(const std::string& str, const std::string& delim, std::vector<std::string>& result) {
  if (delim == "") {
    result.push_back(str);
    return;
  }

  std::string scratch = str;

  while (!scratch.empty()) {
    size_t index = scratch.find(delim);

    // Found a delimeter
    if (index != std::string::npos) {
      // Don't push empty tokens
      if (index > 0) {
        result.push_back(scratch.substr(0, index));
      }

      // Remove the token and the trailing delimeter
      scratch.erase(0, index + delim.size());
    }
    // No delimeters, the rest of the string is a token
    else {
      result.push_back(scratch);

      break;
    }
  }
}

bool ttv::StartsWith(const std::string& str, const std::string& prefix) {
  if (str.size() < prefix.size()) {
    return false;
  }

  return str.substr(0, prefix.size()) == prefix;
}

bool ttv::EndsWith(const std::string& str, const std::string& suffix) {
  if (str.size() < suffix.size()) {
    return false;
  }

  std::string tail = str.substr(str.size() - suffix.size(), suffix.size());

  return tail == suffix;
}

bool ttv::RFC3339TimeToUnixTimestamp(const std::string& str, Timestamp& result) {
  result = 0;

  std::chrono::system_clock::time_point timePoint;
  if (!RFC3339TimeToTimePoint(str, timePoint)) {
    return false;
  }

  result = static_cast<Timestamp>(std::chrono::system_clock::to_time_t(timePoint));

  return true;
}

std::string ttv::UnixTimestampToRFC3339String(Timestamp timestamp) {
  std::ostringstream stringStream;
  std::time_t unixTimestamp = static_cast<time_t>(timestamp);
  std::tm* time = std::gmtime(&unixTimestamp);
  stringStream << std::put_time(time, "%Y-%m-%dT%H:%M:%SZ");
  return stringStream.str();
}

/**
 * A better string copying function which guarantees null termination of the destination and
 * doesn't pad the remainder of dst \0 if src is shorter than maxLen.
 */
void ttv::SafeStringCopy(char* dst, const char* src, size_t maxLen) {
  for (size_t i = 0; i < maxLen; ++i) {
    dst[i] = src[i];

    if (src[i] == '\0') {
      return;
    }
  }

  dst[maxLen - 1] = '\0';
}

std::string ttv::GetGuid() {
  std::uniform_int_distribution<uint64_t> distribution(
    std::numeric_limits<uint64_t>::min(), std::numeric_limits<uint64_t>::max());

  std::array<uint64_t, 2> generated;
  generated[0] = distribution(ttv::random::GetGenerator());
  generated[1] = distribution(ttv::random::GetGenerator());

  const uint8_t* bytes = reinterpret_cast<const uint8_t*>(generated.data());

  std::ostringstream uuidStream;
  uuidStream << std::hex;
  PrintByte(uuidStream, bytes[0]);
  PrintByte(uuidStream, bytes[1]);
  PrintByte(uuidStream, bytes[2]);
  PrintByte(uuidStream, bytes[3]);
  uuidStream << "-";
  PrintByte(uuidStream, bytes[4]);
  PrintByte(uuidStream, bytes[5]);
  uuidStream << "-";
  PrintByte(uuidStream, bytes[6]);
  PrintByte(uuidStream, bytes[7]);
  uuidStream << "-";
  PrintByte(uuidStream, bytes[8]);
  PrintByte(uuidStream, bytes[9]);
  uuidStream << "-";
  PrintByte(uuidStream, bytes[10]);
  PrintByte(uuidStream, bytes[11]);
  PrintByte(uuidStream, bytes[12]);
  PrintByte(uuidStream, bytes[13]);
  PrintByte(uuidStream, bytes[14]);
  PrintByte(uuidStream, bytes[15]);

  return uuidStream.str();
}

/**
 * Parse arguments for command-line samples. Essentially, splits on spaces while treating
 * unescaped quoted segments as complete tokens.
 */
std::vector<std::string> ttv::ParseArguments(std::string args) {
  std::vector<std::string> out;
  unsigned int tokenStart = 0;
  unsigned int currentIndex = 0;

  for (;;) {
    // Advance to beginning of next token.
    while (currentIndex < args.length() && ttv::IsWhitespace(args[currentIndex])) {
      currentIndex++;
    }

    if (currentIndex >= args.length()) {
      break;
    }

    bool tokenStartsWithQuote = args[currentIndex] == '"';

    if (tokenStartsWithQuote) {
      // If we start with a quote, we want to exclude it from the output string.
      currentIndex++;
    }

    tokenStart = currentIndex;

    for (;;) {
      if ((ttv::IsWhitespace(args[currentIndex]) && !tokenStartsWithQuote) || currentIndex >= args.length()) {
        out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
        break;
      }

      bool currentIsUnescapedQuote = args[currentIndex] == '"' && currentIndex > 0 && args[currentIndex - 1] != '\\';
      if (currentIsUnescapedQuote && tokenStartsWithQuote) {
        // End of token formed by quotes
        out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
        break;
      } else if (currentIsUnescapedQuote && !tokenStartsWithQuote) {
        // Token ended by quote in middle of string. Turn back time by one so we treat the
        // quote as the started for the next token.
        out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
        currentIndex--;
        break;
      }

      currentIndex++;
    }

    currentIndex++;
  }

  return out;
}

bool ttv::ParseNum(const std::string& str, int& out) {
  return sscanf(str.c_str(), "%d", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, signed char& out) {
  return sscanf(str.c_str(), "%hhd", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, short int& out) {
  return sscanf(str.c_str(), "%hd", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, long int& out) {
  return sscanf(str.c_str(), "%ld", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, long long int& out) {
  return sscanf(str.c_str(), "%lld", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, unsigned int& out) {
  return sscanf(str.c_str(), "%u", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, unsigned char& out) {
  return sscanf(str.c_str(), "%hhu", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, unsigned short int& out) {
  return sscanf(str.c_str(), "%hu", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, unsigned long int& out) {
  return sscanf(str.c_str(), "%lu", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, unsigned long long int& out) {
  return sscanf(str.c_str(), "%llu", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, float& out) {
  return sscanf(str.c_str(), "%f", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, double& out) {
  return sscanf(str.c_str(), "%lf", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, long double& out) {
  return sscanf(str.c_str(), "%Lf", &out) == 1;
}

bool ttv::ParseNum(const std::string& str, char& out) {
  return sscanf(str.c_str(), "%c", &out) == 1;
}

bool ttv::ParseBool(const std::string& str, bool& out) {
  if (str == "true" || str == "1") {
    out = true;
    return true;
  } else if (str == "false" || str == "0") {
    out = false;
    return true;
  }

  return false;
}
