/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/systemclock.h"

#include <chrono>
#include <ctime>

namespace {
std::shared_ptr<ttv::ISystemClock> gSysClock;

const uint64_t kMillisecondsPerSecond = 1000;
}  // namespace

uint64_t ttv::SystemTimeToMs(uint64_t sysTime) {
  TTV_ASSERT(GetSystemClockFrequency() != 0);
  return sysTime * kMillisecondsPerSecond / GetSystemClockFrequency();
}

uint64_t ttv::MsToSystemTime(uint64_t msTime) {
  return (msTime * GetSystemClockFrequency()) / kMillisecondsPerSecond;
}

uint64_t ttv::GetSystemTimeMilliseconds() {
  return SystemTimeToMs(GetSystemClockTime());
}

ttv::ISystemClock::~ISystemClock() {}

void ttv::SetSystemClock(const std::shared_ptr<ISystemClock>& clock) {
  gSysClock = clock;
}

std::shared_ptr<ttv::ISystemClock> ttv::GetSystemClock() {
  return gSysClock;
}

uint64_t ttv::GetSystemClockFrequency() {
  TTV_ASSERT(gSysClock != nullptr);
  if (gSysClock == nullptr) {
    return 0;
  }

  return gSysClock->GetFrequency();
}

uint64_t ttv::GetSystemClockTime() {
  TTV_ASSERT(gSysClock != nullptr);
  if (gSysClock == nullptr) {
    return 0;
  }

  return gSysClock->GetTime();
}

ttv::Timestamp ttv::GetCurrentTimeAsUnixTimestamp() {
  return static_cast<Timestamp>(std::chrono::seconds(std::time(nullptr)).count());
}
