/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/getchanneltask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/internal/graphql/generated/getchannelqueryinfo.h"
#include "twitchsdk/core/internal/graphql/utilities/channelinfo.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::GetChannelTask::GetChannelTask(ChannelId channelId, Callback callback)
    : mCallback(callback), mChannelId(channelId) {
  TTV_ASSERT(channelId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GetChannelTask created");
}

void ttv::GetChannelTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;

  root["query"] = core::graphql::GetChannelQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["userID"] = std::to_string(mChannelId);

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::GetChannelTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetChannelTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    core::graphql::GetChannelQueryInfo::PayloadType payload;

    if (ttv::json::ToObject(jsonVal["data"], payload)) {
      mResult = std::make_shared<Result>();
      mResult->channelInfo = graphql::GQLUserToChannelInfo(payload);

      auto jUser = jsonVal["data"]["user"];
      if (!jUser.isNull() && jUser.isObject()) {
        ttv::Timestamp value;
        if (ttv::json::DateSchema::Parse(jUser["createdAt"], value)) {
          mResult->channelInfo.createdAtTimestamp = value;
        }

        if (ttv::json::DateSchema::Parse(jUser["updatedAt"], value)) {
          mResult->channelInfo.updatedAtTimestamp = value;
        }
      }
    } else {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::GetChannelTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
