/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/getprofileimageurltask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/types/coretypes.h"

#include <sstream>

namespace {
const char* kHostName = "https://api.twitch.tv";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X 'POST' -H 'Content-Type: application/json' -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID:
// $Client-ID' -H 'Authorization: OAuth $OAuth-Token' -d '{"image_type":"profile_image"}'
// 'https://api.twitch.tv/kraken/users/<UserId>/upload_image' POST /v5/users/:id/upload_image

ttv::GetProfileImageUrlTask::GetProfileImageUrlTask(UserId userId, const std::string& oauthToken, Callback callback)
    : HttpTask(nullptr, nullptr, oauthToken.c_str()), mCallback(callback), mUserId(userId) {
  TTV_ASSERT(userId > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GetProfileImageUrlTask created");
}

void ttv::GetProfileImageUrlTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << "/kraken/users/" << mUserId << "/upload_image";

  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.requestHeaders.emplace_back("Content-Type", "application/json");

  json::Value root;
  root["image_type"] = json::Value("profile_image");

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.url = ss.str();
  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::GetProfileImageUrlTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GetProfileImageUrlTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();

    const auto& jUrl = jsonVal["upload_url"];
    if (!jUrl.isNull() && jUrl.isString()) {
      mResult->uploadUrl = jUrl.asString();
    }

    const auto& jUploadId = jsonVal["upload_id"];
    if (!jUploadId.isNull() && jUploadId.isString()) {
      mResult->uploadId = jUploadId.asString();
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::GetProfileImageUrlTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
