/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/getstreamtask.h"

#include "twitchsdk/core/internal/graphql/generated/getchannelqueryinfo.h"
#include "twitchsdk/core/internal/graphql/generated/getstreamqueryinfo.h"
#include "twitchsdk/core/internal/graphql/utilities/streaminfo.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::GetStreamTask::GetStreamTask(ChannelId channelId, const std::string& oauthToken, Callback callback)
    : HttpTask(oauthToken), mCallback(callback), mChannelId(channelId) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GetStreamTask by channel id created");
}

ttv::GetStreamTask::GetStreamTask(const std::string& channelName, const std::string& oauthToken, Callback callback)
    : HttpTask(oauthToken), mCallback(callback), mChannelName(channelName), mChannelId(0) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GetStreamTask by channel name created");
}

void ttv::GetStreamTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;

  root["query"] = core::graphql::GetStreamQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["userID"] = std::to_string(mChannelId);

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::GetStreamTask::ProcessResponse(uint /*status*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    ttv::json::Value root;

    bool parsed = ParseDocument(response, root);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GetStreamTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (!parsed || root.isNull() || !root.isObject()) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    } else {
      if (root["data"].isNull() || root["data"]["user"].isNull()) {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else if (root["data"]["user"]["stream"].isNull()) {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_NO_STREAMINFO;
      } else {
        core::graphql::GetStreamQueryInfo::PayloadType payload;

        if (ttv::json::ToObject(root["data"], payload)) {
          mResult = std::make_shared<Result>();
          mResult->streamInfo = std::make_unique<StreamInfo>(graphql::GQLUserToStreamInfo(payload));

          ttv::Timestamp value;
          if (ttv::json::DateSchema::Parse(root["data"]["user"]["stream"]["createdAt"], value)) {
            mResult->streamInfo->createdAtTimestamp = value;
          }
        } else {
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        }
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::GetStreamTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
