/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/getusertask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/internal/graphql/generated/getcurrentuserqueryinfo.h"
#include "twitchsdk/core/internal/graphql/generated/getuserbyidqueryinfo.h"
#include "twitchsdk/core/internal/graphql/generated/getuserbyloginqueryinfo.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::GetUserTask::GetUserTask(const std::string& username, Callback callback)
    : mCallback(callback), mUserName(username), mUserId(0) {
  TTV_ASSERT(username.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetUserTask created with username");
}

ttv::GetUserTask::GetUserTask(const std::shared_ptr<OAuthToken>& authToken, Callback callback)
    : ttv::HttpTask(authToken->GetToken()), mCallback(callback), mUserId(0) {
  TTV_ASSERT(authToken != nullptr);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetUserTask created with auth token");
}

ttv::GetUserTask::GetUserTask(UserId userId, Callback callback) : mCallback(callback), mUserId(userId) {
  TTV_ASSERT(userId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetUserTask created with user id");
}

void ttv::GetUserTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);

  if (!mUserName.empty()) {
    root["query"] = core::graphql::GetUserByLoginQueryInfo::kQuery;
    root["variables"]["login"] = mUserName;
  } else if (mUserId != 0) {
    root["query"] = core::graphql::GetUserByIdQueryInfo::kQuery;
    root["variables"]["userID"] = std::to_string(mUserId);
  } else {
    root["query"] = core::graphql::GetCurrentUserQueryInfo::kQuery;
  }

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::GetUserTask::ProcessResponse(uint statusCode, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetUserTask::ProcessResponse - JSON parsing failed");
      PopulateErrorDetails(TTV_EC_WEBAPI_RESULT_INVALID_JSON, statusCode, "JSON parsing failed");
      return;
    }

    mResult = std::make_shared<Result>();
    ttv::json::Value user = ttv::json::Value::null;

    if (!mUserName.empty() || mUserId != 0) {
      if (!jsonVal["data"]["user"].isNull()) {
        user = jsonVal["data"]["user"];
      }
    } else {
      if (!jsonVal["data"]["currentUser"].isNull()) {
        user = jsonVal["data"]["currentUser"];
      }
    }

    if (!user.isNull()) {
      if (!user["login"].isNull()) {
        mResult->userInfo.userName = user["login"].asString();
      }

      if (!user["displayName"].isNull()) {
        mResult->userInfo.displayName = user["displayName"].asString();
      }

      if (!user["description"].isNull()) {
        mResult->userInfo.bio = user["description"].asString();
      }

      if (!user["profileImageURL"].isNull()) {
        mResult->userInfo.logoImageUrl = user["profileImageURL"].asString();
      }

      mResult->userInfo.userId = static_cast<ttv::UserId>(atoi(user["id"].asString().c_str()));

      if (!user["createdAt"].isNull()) {
        RFC3339TimeToUnixTimestamp(user["createdAt"].asString(), mResult->userInfo.createdTimestamp);
      }
    } else {
      PopulateErrorDetails(TTV_EC_WEBAPI_RESULT_INVALID_JSON, statusCode, "null user");
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    PopulateErrorDetails(TTV_EC_WEBAPI_RESULT_INVALID_JSON, statusCode, "empty body");
  }
}

void ttv::GetUserTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus, mResult);
  }
}
