/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/task/getvodtask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/internal/graphql/generated/getvodqueryinfo.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/types/coretypes.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::GetVodTask::GetVodTask(const std::string& vodId, Callback callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(callback), mVodId(vodId) {
  TTV_ASSERT(vodId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GetVodTask created");
}

void ttv::GetVodTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;
  root["query"] = core::graphql::GetVodQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["id"] = mVodId;

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::GetVodTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside GetVodTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();

    core::graphql::GetVodQueryInfo::PayloadType payload;

    if (ttv::json::ToObject(jsonVal["data"], payload)) {
      mResult->vodId = payload.id;
      mResult->title = payload.title.ValueOrDefault("");

      if (payload.owner.HasValue()) {
        mResult->channelId = static_cast<ttv::ChannelId>(std::strtoul(payload.owner.Value().id.c_str(), nullptr, 10));
      }

      mResult->vodType = VodType::Unknown;

      if (payload.broadcastType.HasValue()) {
        auto broadcastType = payload.broadcastType.Value();

        if (broadcastType == core::graphql::GetVodQueryInfo::BroadcastType::ARCHIVE) {
          mResult->vodType = VodType::Archive;
        } else if (broadcastType == core::graphql::GetVodQueryInfo::BroadcastType::HIGHLIGHT) {
          mResult->vodType = VodType::Highlight;
        } else if (broadcastType == core::graphql::GetVodQueryInfo::BroadcastType::UPLOAD) {
          mResult->vodType = VodType::Upload;
        }
      }

      mResult->status = VodStatus::Unknown;

      if (payload.status.HasValue()) {
        auto status = payload.status.Value();

        if (status == core::graphql::GetVodQueryInfo::VideoStatus::RECORDED) {
          mResult->status = VodStatus::Recorded;
        } else if (status == core::graphql::GetVodQueryInfo::VideoStatus::RECORDING) {
          mResult->status = VodStatus::Recording;
        }
      }

      mResult->durationSeconds = static_cast<uint32_t>(payload.lengthSeconds.ValueOrDefault(0));

      // Manually parse timestamp due to codegen bug
      // Parent nodes known to be non-null because ToObject() succeeded.
      if (!jsonVal["data"]["video"]["recordedAt"].isNull()) {
        ttv::RFC3339TimeToUnixTimestamp(jsonVal["data"]["video"]["recordedAt"].asString(), mResult->recordedAtSeconds);
      }
    } else {
      ttv::trace::Message("GetVodTask", MessageLevel::Error, "Inside GetVodTask::ProcessResponse - ToObject failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::GetVodTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
